<?php

namespace Tests\Feature;

use Tests\TestCase;
use App\Models\Booking;
use App\Models\User;
use App\Models\Vendor;
use App\Models\Property;
use Illuminate\Foundation\Testing\RefreshDatabase;

class PaymentWebhookTest extends TestCase
{
    use RefreshDatabase;

    public function test_paystack_webhook_processes_payment(): void
    {
        $this->seed();
        
        $user = User::factory()->create();
        $vendorUser = User::factory()->create();
        $vendorUser->assignRole('vendor');
        $vendor = Vendor::factory()->create(['user_id' => $vendorUser->id]);
        $property = Property::factory()->create(['vendor_id' => $vendor->id]);

        $booking = Booking::factory()->create([
            'user_id' => $user->id,
            'property_id' => $property->id,
            'vendor_id' => $vendor->id,
            'payment_reference' => 'TEST-REF-12345',
            'total_amount' => 50000,
            'payment_status' => 'pending',
        ]);

        $payload = [
            'event' => 'charge.success',
            'data' => [
                'reference' => 'TEST-REF-12345',
                'amount' => 5000000, // in kobo
                'currency' => 'NGN',
                'status' => 'success',
                'id' => 'trx_123456',
            ],
        ];

        $signature = hash_hmac('sha512', json_encode($payload), config('services.paystack.webhook_secret'));

        $response = $this->postJson(route('webhooks.paystack'), $payload, [
            'X-Paystack-Signature' => $signature,
        ]);

        $response->assertStatus(200);

        $this->assertDatabaseHas('bookings', [
            'id' => $booking->id,
            'payment_status' => 'paid',
            'payment_method' => 'paystack',
        ]);

        $this->assertDatabaseHas('transactions', [
            'booking_id' => $booking->id,
            'reference' => 'TEST-REF-12345',
            'status' => 'completed',
        ]);
    }
}