<?php

namespace Tests\Feature;

use Tests\TestCase;
use App\Models\User;
use App\Models\Vendor;
use App\Models\Property;
use App\Models\Booking;
use Illuminate\Foundation\Testing\RefreshDatabase;

class BookingFlowTest extends TestCase
{
    use RefreshDatabase;

    public function test_user_can_create_booking_for_normal_stay(): void
    {
        $this->seed();
        
        $user = User::factory()->create();
        $user->assignRole('user');
        
        $vendorUser = User::factory()->create();
        $vendorUser->assignRole('vendor');
        $vendor = Vendor::factory()->create(['user_id' => $vendorUser->id]);
        $property = Property::factory()->create([
            'vendor_id' => $vendor->id,
            'max_allowed_guests' => 4,
        ]);

        $response = $this->actingAs($user)->post(route('bookings.store', $property), [
            'start_date' => now()->addDays(2)->format('Y-m-d'),
            'end_date' => now()->addDays(5)->format('Y-m-d'),
            'adults_count' => 2,
            'children_count' => 0,
            'trip_type' => 'normal',
        ]);

        $this->assertDatabaseHas('bookings', [
            'user_id' => $user->id,
            'property_id' => $property->id,
            'requires_vendor_approval' => false,
        ]);
    }

    public function test_party_booking_requires_vendor_approval(): void
    {
        $this->seed();
        
        $user = User::factory()->create();
        $user->assignRole('user');
        
        $vendorUser = User::factory()->create();
        $vendorUser->assignRole('vendor');
        $vendor = Vendor::factory()->create(['user_id' => $vendorUser->id]);
        $property = Property::factory()->create([
            'vendor_id' => $vendor->id,
            'allow_parties' => true,
        ]);

        $response = $this->actingAs($user)->post(route('bookings.store', $property), [
            'start_date' => now()->addDays(2)->format('Y-m-d'),
            'end_date' => now()->addDays(5)->format('Y-m-d'),
            'adults_count' => 2,
            'children_count' => 0,
            'trip_type' => 'party',
        ]);

        $this->assertDatabaseHas('bookings', [
            'user_id' => $user->id,
            'property_id' => $property->id,
            'requires_vendor_approval' => true,
            'vendor_approval_status' => 'pending',
        ]);
    }

    public function test_exceeding_guest_limit_requires_approval(): void
    {
        $this->seed();
        
        $user = User::factory()->create();
        $user->assignRole('user');
        
        $vendorUser = User::factory()->create();
        $vendorUser->assignRole('vendor');
        $vendor = Vendor::factory()->create(['user_id' => $vendorUser->id]);
        $property = Property::factory()->create([
            'vendor_id' => $vendor->id,
            'max_allowed_guests' => 2,
        ]);

        $response = $this->actingAs($user)->post(route('bookings.store', $property), [
            'start_date' => now()->addDays(2)->format('Y-m-d'),
            'end_date' => now()->addDays(5)->format('Y-m-d'),
            'adults_count' => 3,
            'children_count' => 1,
            'trip_type' => 'normal',
        ]);

        $this->assertDatabaseHas('bookings', [
            'user_id' => $user->id,
            'property_id' => $property->id,
            'requires_vendor_approval' => true,
        ]);
    }
}