
<?php

// routes/web.php

use Illuminate\Support\Facades\Route;
use App\Http\Controllers\PropertyController;
use App\Http\Controllers\BookingController;
use App\Http\Controllers\PaymentCallbackController;
use App\Http\Controllers\Auth\VerificationController;
use App\Http\Controllers\ContactController;

Route::get('/', function () {
    $featuredDiscounts = \App\Models\DiscountDeal::where('is_active', true)
        ->where('is_featured', true)
        ->where(function($query) {
            $query->whereNull('valid_from')
                  ->orWhere('valid_from', '<=', now());
        })
        ->where(function($query) {
            $query->whereNull('valid_until')
                  ->orWhere('valid_until', '>=', now());
        })
        ->orderBy('created_at', 'desc')
        ->limit(3)
        ->get();
    
    return view('home', compact('featuredDiscounts'));
})->name('home');

// CAPTCHA
Route::get('/captcha/generate', [\App\Http\Controllers\CaptchaController::class, 'generate'])->name('captcha.generate');

// Language Switcher
Route::get('/language/{code}', [\App\Http\Controllers\LanguageController::class, 'switch'])->name('language.switch');

// Public routes
Route::get('/properties', [PropertyController::class, 'index'])->name('properties.index');
Route::get('/properties/{property}', [PropertyController::class, 'show'])->name('properties.show');

// Car Rental Routes (Public)
Route::get('/cars', [\App\Http\Controllers\CarRentalController::class, 'index'])->name('cars.index');
Route::get('/cars/{car}', [\App\Http\Controllers\CarRentalController::class, 'show'])->name('cars.show');

// Static pages
Route::get('/about', function () {
    return view('pages.about');
})->name('about');

Route::get('/faq', function () {
    return view('pages.faq');
})->name('faq');

Route::get('/contact', function () {
    return view('pages.contact');
})->name('contact');

Route::post('/contact', [ContactController::class, 'submit'])->name('contact.submit');

// Guest Activity Tracking (No auth required)
Route::post('/guest/track-search', [\App\Http\Controllers\GuestActivityController::class, 'trackSearch'])->name('guest.track-search');
Route::post('/guest/track-property/{property}', [\App\Http\Controllers\GuestActivityController::class, 'trackPropertyView'])->name('guest.track-property');
Route::get('/guest/recent-searches', [\App\Http\Controllers\GuestActivityController::class, 'getRecentSearches'])->name('guest.recent-searches');
Route::get('/guest/viewed-properties', [\App\Http\Controllers\GuestActivityController::class, 'getViewedProperties'])->name('guest.viewed-properties');
Route::post('/guest/clear-activity', [\App\Http\Controllers\GuestActivityController::class, 'clearActivity'])->name('guest.clear-activity');

// Auth routes (Laravel Breeze or custom)
require __DIR__.'/auth.php';

Route::middleware(['auth'])->group(function () {
    Route::get('/email/verify', [VerificationController::class, 'show'])
        ->name('verification.notice');
    
    Route::get('/email/verify/{id}/{hash}', [VerificationController::class, 'verify'])
        ->middleware(['signed'])
        ->name('verification.verify');
    
    Route::post('/email/verification-notification', [VerificationController::class, 'resend'])
        ->middleware(['throttle:6,1'])
        ->name('verification.send');
    
    Route::get('/email/verify-manual', [VerificationController::class, 'manual'])
        ->name('verification.manual');
});

// Verified user routes
Route::middleware(['auth', 'verified'])->group(function () {
    // User profile
    Route::get('/profile', [\App\Http\Controllers\User\ProfileController::class, 'edit'])->name('profile');
    Route::put('/profile', [\App\Http\Controllers\User\ProfileController::class, 'update'])->name('profile.update');
    Route::put('/profile/password', [\App\Http\Controllers\User\ProfileController::class, 'updatePassword'])->name('profile.password.update');
    
    // Favorites
    Route::get('/favorites', [\App\Http\Controllers\User\FavoriteController::class, 'index'])->name('favorites.index');
    Route::post('/favorites/toggle/{property}', [\App\Http\Controllers\User\FavoriteController::class, 'toggle'])->name('favorites.toggle');
    
    // Bookings
    Route::get('/bookings', [BookingController::class, 'index'])->name('bookings.index');
    Route::get('/bookings/create/{property}', [BookingController::class, 'create'])->name('bookings.create');
    Route::post('/bookings/{property}', [BookingController::class, 'store'])->name('bookings.store');
    Route::get('/bookings/{booking}', [BookingController::class, 'show'])->name('bookings.show');
    Route::get('/bookings/{booking}/payment', [BookingController::class, 'payment'])->name('bookings.payment');
    Route::post('/bookings/{booking}/payment', [BookingController::class, 'processPayment'])->name('bookings.process-payment');
    Route::get('/bookings/{booking}/bank-transfer', [BookingController::class, 'bankTransferInstructions'])->name('bookings.bank-transfer-instructions');
    Route::post('/bookings/{booking}/upload-proof', [BookingController::class, 'uploadPaymentProof'])->name('bookings.upload-payment-proof');
    Route::post('/bookings/{booking}/confirm-checkin', [BookingController::class, 'confirmCheckIn'])->name('bookings.confirm-checkin');
    Route::post('/bookings/{booking}/confirm-checkout', [BookingController::class, 'confirmCheckOut'])->name('bookings.confirm-checkout');
    Route::post('/bookings/{booking}/request-refund', [BookingController::class, 'requestRefund'])->name('bookings.request-refund');
    Route::get('/bookings/{booking}/receipt', [BookingController::class, 'downloadReceipt'])->name('bookings.receipt');
    Route::post('/bookings/{booking}/complete', [BookingController::class, 'complete'])->name('bookings.complete');
    
    // Property Reviews
    Route::get('/reviews/create', [\App\Http\Controllers\ReviewController::class, 'create'])->name('reviews.create');
    Route::post('/reviews', [\App\Http\Controllers\ReviewController::class, 'store'])->name('reviews.store');
    
    // User Transactions
    Route::get('/transactions', [\App\Http\Controllers\User\TransactionController::class, 'index'])->name('transactions.index');
    
    // Car Rental Bookings
    Route::get('/cars/{car}/booking', [\App\Http\Controllers\CarRentalController::class, 'booking'])->name('cars.booking');
    Route::post('/cars/{car}/booking', [\App\Http\Controllers\CarRentalController::class, 'storeBooking'])->name('cars.booking.store');
    Route::get('/car-bookings/{booking}', [\App\Http\Controllers\CarRentalController::class, 'showBooking'])->name('cars.booking.show');
    Route::get('/car-bookings/{booking}/payment', [\App\Http\Controllers\CarRentalController::class, 'payment'])->name('cars.booking.payment');
    Route::post('/car-bookings/{booking}/payment', [\App\Http\Controllers\CarRentalController::class, 'processPayment'])->name('cars.booking.process-payment');
    Route::get('/car-bookings/{booking}/bank-transfer', [\App\Http\Controllers\CarRentalController::class, 'bankTransferInstructions'])->name('cars.booking.bank-transfer');
    Route::post('/car-bookings/{booking}/upload-proof', [\App\Http\Controllers\CarRentalController::class, 'uploadPaymentProof'])->name('cars.booking.upload-payment-proof');
    Route::post('/car-bookings/{booking}/confirm-pickup', [\App\Http\Controllers\CarRentalController::class, 'confirmPickup'])->name('cars.booking.confirm-pickup');
    Route::post('/car-bookings/{booking}/confirm-return', [\App\Http\Controllers\CarRentalController::class, 'confirmReturn'])->name('cars.booking.confirm-return');
    Route::post('/car-bookings/{booking}/request-refund', [\App\Http\Controllers\CarRentalController::class, 'requestRefund'])->name('cars.booking.request-refund');
    // Car Reviews
    Route::get('/car-bookings/{booking}/review', [\App\Http\Controllers\CarReviewController::class, 'create'])->name('cars.reviews.create');
    Route::post('/car-bookings/{booking}/review', [\App\Http\Controllers\CarReviewController::class, 'store'])->name('cars.reviews.store');
    
    // Nearby attractions (unlocked after payment)
    Route::get('/properties/{property}/nearby', [PropertyController::class, 'showNearbyAttractions'])
        ->name('properties.nearby-attractions');
});

// Payment callback
Route::get('/payment/callback', [PaymentCallbackController::class, 'callback'])->name('payment.callback');

// Vendor registration (public)
Route::get('/vendor/register', function () {
    return view('auth.vendor-register');
})->middleware('guest')->name('vendor.register');

Route::get('/vendor/login', function () {
    return redirect()->route('login');
})->name('vendor.login');

// Vendor routes
Route::prefix('vendor')->name('vendor.')->middleware(['auth', 'role:vendor'])->group(function () {
    Route::get('/dashboard', [\App\Http\Controllers\Vendor\DashboardController::class, 'index'])->name('dashboard');
    
    // Transactions
    Route::get('/transactions', [\App\Http\Controllers\Vendor\TransactionController::class, 'index'])->name('transactions.index');
    
    // Profile
    Route::get('/profile', [\App\Http\Controllers\Vendor\ProfileController::class, 'edit'])->name('profile');
    Route::put('/profile', [\App\Http\Controllers\Vendor\ProfileController::class, 'update'])->name('profile.update');
    Route::put('/profile/password', [\App\Http\Controllers\Vendor\ProfileController::class, 'updatePassword'])->name('profile.password.update');
    Route::put('/profile/pin', [\App\Http\Controllers\Vendor\ProfileController::class, 'updatePin'])->name('profile.pin.update');
    
    // KYC
    Route::get('/kyc', [\App\Http\Controllers\Vendor\KycController::class, 'show'])->name('kyc.show');
    Route::post('/kyc', [\App\Http\Controllers\Vendor\KycController::class, 'submit'])->name('kyc.submit');
    
    Route::resource('properties', \App\Http\Controllers\Vendor\PropertyController::class);
    
    // Car Rental Management
    Route::resource('cars', \App\Http\Controllers\Vendor\CarController::class);
    Route::delete('/cars/{car}/photos/{index}', [\App\Http\Controllers\Vendor\CarController::class, 'deletePhoto'])->name('cars.photos.delete');
    
    Route::get('/bookings', [\App\Http\Controllers\Vendor\BookingController::class, 'index'])->name('bookings.index');
    Route::get('/bookings/{booking}', [\App\Http\Controllers\Vendor\BookingController::class, 'show'])->name('bookings.show');
    Route::post('/bookings/{booking}/approve', [\App\Http\Controllers\Vendor\BookingController::class, 'approve'])->name('bookings.approve');
    Route::post('/bookings/{booking}/reject', [\App\Http\Controllers\Vendor\BookingController::class, 'reject'])->name('bookings.reject');
    Route::post('/bookings/{booking}/caution', [\App\Http\Controllers\Vendor\BookingController::class, 'decideCautionRefund'])->name('bookings.caution');
    
    // Car Bookings
    Route::get('/car-bookings/{booking}', [\App\Http\Controllers\Vendor\BookingController::class, 'showCarBooking'])->name('car-bookings.show');
    Route::post('/car-bookings/{booking}/report-damage', [\App\Http\Controllers\Vendor\BookingController::class, 'reportDamage'])->name('car-bookings.report-damage');
    
    Route::get('/ads', [\App\Http\Controllers\Vendor\AdController::class, 'index'])->name('ads.index');
    Route::post('/ads', [\App\Http\Controllers\Vendor\AdController::class, 'store'])->name('ads.store');
    Route::get('/ads/{vendorAd}/payment', [\App\Http\Controllers\Vendor\AdController::class, 'showPayment'])->name('ads.payment');
    Route::post('/ads/{vendorAd}/payment', [\App\Http\Controllers\Vendor\AdController::class, 'processPayment'])->name('ads.process-payment');
    Route::get('/ads/{vendorAd}/bank-transfer', [\App\Http\Controllers\Vendor\AdController::class, 'showBankTransfer'])->name('ads.bank-transfer');
    Route::post('/ads/{vendorAd}/upload-proof', [\App\Http\Controllers\Vendor\AdController::class, 'uploadProof'])->name('ads.upload-proof');
    
    Route::get('/payouts', [\App\Http\Controllers\Vendor\PayoutController::class, 'index'])->name('payouts.index');
    Route::post('/payouts', [\App\Http\Controllers\Vendor\PayoutController::class, 'request'])->name('payouts.request');
});

// Admin routes
Route::prefix('admin')->name('admin.')->middleware(['auth', 'role:owner|manager|moderator'])->group(function () {
    Route::get('/dashboard', [\App\Http\Controllers\Admin\DashboardController::class, 'index'])->name('dashboard');
    
    // All Users (Owner, Manager only)
    Route::get('/all-users', [\App\Http\Controllers\Admin\AllUsersController::class, 'index'])
        ->middleware('role:owner|manager')->name('all-users.index');
    Route::get('/all-users/{user}', [\App\Http\Controllers\Admin\AllUsersController::class, 'show'])
        ->middleware('role:owner|manager')->name('all-users.show');
    
    // Bookings
    Route::get('/bookings', [\App\Http\Controllers\Admin\BookingController::class, 'index'])->name('bookings.index');
    Route::get('/bookings/{booking}', [\App\Http\Controllers\Admin\BookingController::class, 'show'])->name('bookings.show');
    Route::post('/bookings/{booking}/verify', [\App\Http\Controllers\Admin\BookingController::class, 'verifyPayment'])
        ->middleware('role:owner|manager')->name('bookings.verify');
    Route::post('/bookings/{booking}/caution-override', [\App\Http\Controllers\Admin\BookingController::class, 'overrideCautionRefund'])
        ->middleware('role:owner|manager')->name('bookings.caution-override');
    Route::post('/bookings/{booking}/resolve-caution', [\App\Http\Controllers\Admin\BookingController::class, 'resolveCaution'])
        ->middleware('role:owner|manager')->name('bookings.resolve-caution');
    Route::post('/bookings/{booking}/process-refund', [\App\Http\Controllers\Admin\BookingController::class, 'processRefund'])
        ->middleware('role:owner|manager')->name('bookings.process-refund');
    Route::post('/bookings/{booking}/cancel', [\App\Http\Controllers\Admin\BookingController::class, 'cancel'])
        ->middleware('role:owner|manager')->name('bookings.cancel');
    
    // Transactions & Payouts
    Route::get('/transactions', [\App\Http\Controllers\Admin\TransactionController::class, 'index'])->name('transactions.index');
    Route::get('/refund-payouts', [\App\Http\Controllers\Admin\TransactionController::class, 'payouts'])->name('refund-payouts.index');
    
    // Vendors
    Route::get('/vendors', [\App\Http\Controllers\Admin\VendorController::class, 'index'])->name('vendors.index');
    Route::get('/vendors/{vendor}', [\App\Http\Controllers\Admin\VendorController::class, 'show'])->name('vendors.show');
    Route::post('/vendors/{vendor}/verify', [\App\Http\Controllers\Admin\VendorController::class, 'verify'])->name('vendors.verify');
    Route::post('/vendors/{vendor}/suspend', [\App\Http\Controllers\Admin\VendorController::class, 'suspend'])->name('vendors.suspend');
    Route::post('/vendors/{vendor}/reset-pin', [\App\Http\Controllers\Admin\VendorController::class, 'resetPin'])->name('vendors.reset-pin');
    
   // KYC Approvals
Route::get('/kyc', [\App\Http\Controllers\Admin\KycApprovalController::class, 'index'])
    ->middleware('role:owner|manager')->name('kyc.index');
Route::post('/kyc/{vendor}/approve', [\App\Http\Controllers\Admin\KycApprovalController::class, 'approve'])
    ->middleware('role:owner|manager')->name('kyc.approve');
Route::post('/kyc/{vendor}/reject', [\App\Http\Controllers\Admin\KycApprovalController::class, 'reject'])
    ->middleware('role:owner|manager')->name('kyc.reject');
Route::post('/kyc/{vendor}/verify-business', [\App\Http\Controllers\Admin\KycApprovalController::class, 'verifyBusiness'])
    ->middleware('role:owner|manager')->name('kyc.verify-business');
    
    // KYC Form Builder
    Route::resource('kyc-form', \App\Http\Controllers\Admin\KycFormBuilderController::class)
        ->middleware('role:owner|manager');
    Route::post('/kyc-form/reorder', [\App\Http\Controllers\Admin\KycFormBuilderController::class, 'reorder'])
        ->middleware('role:owner|manager')->name('kyc-form.reorder');
    
    // Property Management (Admin can create, edit, delete)
    Route::get('/properties', [\App\Http\Controllers\Admin\PropertyApprovalController::class, 'index'])->name('properties.index');
    Route::get('/properties/create', [\App\Http\Controllers\Admin\PropertyController::class, 'create'])
        ->middleware('role:owner|manager')->name('properties.create');
    Route::post('/properties', [\App\Http\Controllers\Admin\PropertyController::class, 'store'])
        ->middleware('role:owner|manager')->name('properties.store');
    Route::get('/properties/{property}', [\App\Http\Controllers\Admin\PropertyApprovalController::class, 'show'])->name('properties.show');
    Route::get('/properties/{property}/edit', [\App\Http\Controllers\Admin\PropertyController::class, 'edit'])
        ->middleware('role:owner|manager')->name('properties.edit');
    Route::put('/properties/{property}', [\App\Http\Controllers\Admin\PropertyController::class, 'update'])
        ->middleware('role:owner|manager')->name('properties.update');
    Route::delete('/properties/{property}', [\App\Http\Controllers\Admin\PropertyController::class, 'destroy'])
        ->middleware('role:owner|manager')->name('properties.destroy');
    Route::post('/properties/{property}/approve', [\App\Http\Controllers\Admin\PropertyApprovalController::class, 'approve'])
        ->middleware('role:owner|manager')->name('properties.approve');
    Route::post('/properties/{property}/reject', [\App\Http\Controllers\Admin\PropertyApprovalController::class, 'reject'])
        ->middleware('role:owner|manager')->name('properties.reject');
    Route::post('/properties/{property}/toggle', [\App\Http\Controllers\Admin\PropertyApprovalController::class, 'toggleStatus'])
        ->middleware('role:owner|manager')->name('properties.toggle');
    
    // Advertisement Approvals
    Route::get('/ads', [\App\Http\Controllers\Admin\VendorAdController::class, 'index'])->name('ads.index');
    Route::post('/ads/{vendorAd}/verify-payment', [\App\Http\Controllers\Admin\VendorAdController::class, 'verifyPayment'])
        ->middleware('role:owner|manager')->name('ads.verify-payment');
    Route::post('/ads/{vendorAd}/reject-payment', [\App\Http\Controllers\Admin\VendorAdController::class, 'rejectPayment'])
        ->middleware('role:owner|manager')->name('ads.reject-payment');
    Route::post('/ads/{vendorAd}/approve', [\App\Http\Controllers\Admin\VendorAdController::class, 'approve'])
        ->middleware('role:owner|manager')->name('ads.approve');
    Route::post('/ads/{vendorAd}/deactivate', [\App\Http\Controllers\Admin\VendorAdController::class, 'deactivate'])
        ->middleware('role:owner|manager')->name('ads.deactivate');
    Route::post('/ads/{vendorAd}/extend', [\App\Http\Controllers\Admin\VendorAdController::class, 'extend'])
        ->middleware('role:owner|manager')->name('ads.extend');
    
    // Car Rental Management
    Route::get('/cars', [\App\Http\Controllers\Admin\CarController::class, 'index'])->name('cars.index');
    Route::get('/cars/{car}', [\App\Http\Controllers\Admin\CarController::class, 'show'])->name('cars.show');
    Route::post('/cars/{car}/approve', [\App\Http\Controllers\Admin\CarController::class, 'approve'])
        ->middleware('role:owner|manager')->name('cars.approve');
    Route::post('/cars/{car}/reject', [\App\Http\Controllers\Admin\CarController::class, 'reject'])
        ->middleware('role:owner|manager')->name('cars.reject');
    Route::delete('/cars/{car}', [\App\Http\Controllers\Admin\CarController::class, 'destroy'])
        ->middleware('role:owner|manager')->name('cars.destroy');
    
    // Car Bookings Management
    Route::get('/car-bookings', [\App\Http\Controllers\Admin\CarBookingController::class, 'index'])->name('car-bookings.index');
    Route::get('/car-bookings/{booking}', [\App\Http\Controllers\Admin\CarBookingController::class, 'show'])->name('car-bookings.show');
    Route::post('/car-bookings/{booking}/status', [\App\Http\Controllers\Admin\CarBookingController::class, 'updateStatus'])
        ->middleware('role:owner|manager')->name('car-bookings.update-status');
    Route::post('/car-bookings/{booking}/verify-payment', [\App\Http\Controllers\Admin\CarBookingController::class, 'verifyPayment'])
        ->middleware('role:owner|manager')->name('car-bookings.verify-payment');
    Route::post('/car-bookings/{booking}/resolve-deposit', [\App\Http\Controllers\Admin\CarBookingController::class, 'resolveDeposit'])
        ->middleware('role:owner|manager')->name('car-bookings.resolve-deposit');
        Route::post('/car-bookings/{booking}/process-refund', [\App\Http\Controllers\Admin\CarBookingController::class, 'processRefund'])
        ->middleware('role:owner|manager')->name('car-bookings.process-refund');
    
    // Car Reviews Management
    Route::post('/car-reviews/{review}/approve', [\App\Http\Controllers\Admin\CarBookingController::class, 'approveReview'])
        ->middleware('role:owner|manager')->name('car-reviews.approve');
    Route::post('/car-reviews/{review}/reject', [\App\Http\Controllers\Admin\CarBookingController::class, 'rejectReview'])
        ->middleware('role:owner|manager')->name('car-reviews.reject');
    
    // Admin Users (Owner only)
    Route::get('/users', [\App\Http\Controllers\Admin\UserController::class, 'index'])
        ->middleware('role:owner')->name('users.index');
    Route::post('/users', [\App\Http\Controllers\Admin\UserController::class, 'store'])
        ->middleware('role:owner')->name('users.store');
    Route::put('/users/{user}', [\App\Http\Controllers\Admin\UserController::class, 'update'])
        ->middleware('role:owner')->name('users.update');
    Route::delete('/users/{user}', [\App\Http\Controllers\Admin\UserController::class, 'destroy'])
        ->middleware('role:owner')->name('users.destroy');
    Route::post('/users/{user}/toggle', [\App\Http\Controllers\Admin\UserController::class, 'toggleStatus'])
        ->middleware('role:owner')->name('users.toggle');
    
    // Payment Verification (Owner, Manager, Moderator)
    Route::get('/payment-verification', [\App\Http\Controllers\Admin\PaymentVerificationController::class, 'index'])
        ->middleware('role:owner|manager|moderator')->name('payment-verification.index');
    Route::post('/payment-verification/{booking}/approve', [\App\Http\Controllers\Admin\PaymentVerificationController::class, 'approve'])
        ->middleware('role:owner|manager|moderator')->name('payment-verification.approve');
    Route::post('/payment-verification/{booking}/reject', [\App\Http\Controllers\Admin\PaymentVerificationController::class, 'reject'])
        ->middleware('role:owner|manager|moderator')->name('payment-verification.reject');
    
    // Vendor Payouts (Owner, Manager)
    Route::get('/payouts', [\App\Http\Controllers\Admin\PayoutController::class, 'index'])
        ->middleware('role:owner|manager')->name('payouts.index');
    Route::post('/payouts/{payout}/approve', [\App\Http\Controllers\Admin\PayoutController::class, 'approve'])
        ->middleware('role:owner|manager')->name('payouts.approve');
    Route::post('/payouts/{payout}/reject', [\App\Http\Controllers\Admin\PayoutController::class, 'reject'])
        ->middleware('role:owner|manager')->name('payouts.reject');
    
    // Payment Methods (Owner only)
    Route::get('/payment-methods', [\App\Http\Controllers\Admin\PaymentMethodController::class, 'index'])
        ->middleware('role:owner')->name('payment-methods.index');
    Route::put('/payment-methods/{paymentMethod}', [\App\Http\Controllers\Admin\PaymentMethodController::class, 'update'])
        ->middleware('role:owner')->name('payment-methods.update');
    Route::post('/payment-methods/{paymentMethod}/toggle', [\App\Http\Controllers\Admin\PaymentMethodController::class, 'toggleStatus'])
        ->middleware('role:owner')->name('payment-methods.toggle');
    
    // Ad Plans (Owner, Manager)
    Route::get('/ad-plans', [\App\Http\Controllers\Admin\AdPlanController::class, 'index'])
        ->middleware('role:owner|manager')->name('ad-plans.index');
    Route::post('/ad-plans', [\App\Http\Controllers\Admin\AdPlanController::class, 'store'])
        ->middleware('role:owner|manager')->name('ad-plans.store');
    Route::put('/ad-plans/{adPlan}', [\App\Http\Controllers\Admin\AdPlanController::class, 'update'])
        ->middleware('role:owner|manager')->name('ad-plans.update');
    Route::delete('/ad-plans/{adPlan}', [\App\Http\Controllers\Admin\AdPlanController::class, 'destroy'])
        ->middleware('role:owner|manager')->name('ad-plans.destroy');
    Route::post('/ad-plans/{adPlan}/toggle', [\App\Http\Controllers\Admin\AdPlanController::class, 'toggleStatus'])
        ->middleware('role:owner|manager')->name('ad-plans.toggle');
    
    
    // Promo Codes (Owner, Manager)
    Route::resource('promo-codes', \App\Http\Controllers\Admin\PromoCodeController::class)
        ->middleware('role:owner|manager');
    Route::post('/promo-codes/{promoCode}/toggle', [\App\Http\Controllers\Admin\PromoCodeController::class, 'toggle'])
        ->middleware('role:owner|manager')->name('promo-codes.toggle');
    
    // Discount Deals (Owner, Manager)
    Route::resource('discount-deals', \App\Http\Controllers\Admin\DiscountDealController::class)
        ->middleware('role:owner|manager');
    Route::post('/discount-deals/{discountDeal}/toggle', [\App\Http\Controllers\Admin\DiscountDealController::class, 'toggle'])
        ->middleware('role:owner|manager')->name('discount-deals.toggle');
    Route::post('/discount-deals/{discountDeal}/toggle-featured', [\App\Http\Controllers\Admin\DiscountDealController::class, 'toggleFeatured'])
        ->middleware('role:owner|manager')->name('discount-deals.toggle-featured');
    
    // Popular Destinations (Owner, Manager)
    Route::resource('destinations', \App\Http\Controllers\Admin\PopularDestinationController::class)
        ->middleware('role:owner|manager');
    
    // House Type Images (Owner, Manager)
    Route::resource('house-types', \App\Http\Controllers\Admin\HouseTypeImageController::class)
        ->middleware('role:owner|manager');
    
    // FAQs (Owner, Manager)
    Route::resource('faqs', \App\Http\Controllers\Admin\FaqController::class)
        ->middleware('role:owner|manager');
    Route::patch('/faqs/{faq}/toggle', [\App\Http\Controllers\Admin\FaqController::class, 'toggle'])
        ->middleware('role:owner|manager')->name('faqs.toggle');
    
    // Settings (Owner only)
    Route::get('/settings', [\App\Http\Controllers\Admin\SettingsController::class, 'index'])
        ->middleware('role:owner')->name('settings');
    Route::post('/settings', [\App\Http\Controllers\Admin\SettingsController::class, 'update'])
        ->middleware('role:owner')->name('settings.update');
    
    // Audit Logs (Owner only)
    Route::get('/audit-logs', [\App\Http\Controllers\Admin\AuditLogController::class, 'index'])
        ->middleware('role:owner')->name('audit-logs');

        
});

// API validation endpoints (web authenticated)
Route::middleware('auth')->group(function () {
    // Discount code validation (for users)
    Route::post('/api/v1/discount-codes/validate', [\App\Http\Controllers\Api\DiscountCodeController::class, 'validate']);
    
    // Promo code validation (for vendors)
    Route::post('/api/v1/promo-codes/validate', [\App\Http\Controllers\Api\PromoCodeController::class, 'validate']);
});