<?php

use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\Auth;
use Illuminate\Http\Request;
use App\Models\User;
use App\Models\Vendor;
use App\Models\Setting;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rules\Password;
use Illuminate\Support\Facades\Password as PasswordFacade;
use Illuminate\Auth\Events\PasswordReset;
use Illuminate\Support\Str;

// Show login form
Route::get('/login', function () {
    return view('auth.login');
})->middleware('guest')->name('login');

// Handle login
Route::post('/login', function (Request $request) {
    $credentials = $request->validate([
        'email' => ['required', 'email'],
        'password' => ['required'],
    ]);

    if (Auth::attempt($credentials, $request->boolean('remember'))) {
        $request->session()->regenerate();
        
        $user = Auth::user();
        
        // Redirect based on user role
        if ($user->hasAnyRole(['owner', 'manager', 'moderator'])) {
            return redirect()->intended(route('admin.dashboard'));
        } elseif ($user->hasRole('vendor')) {
            return redirect()->intended(route('vendor.dashboard'));
        }
        
        // Regular user
        return redirect()->intended('/');
    }

    return back()->withErrors([
        'email' => 'The provided credentials do not match our records.',
    ])->onlyInput('email');
})->middleware('guest');

// Show registration form
Route::get('/register', function () {
    // Check if user or vendor registration is enabled
    $userRegistrationEnabled = Setting::get('enable_user_registration', true);
    $vendorRegistrationEnabled = Setting::get('enable_vendor_registration', true);
    
    // If both are disabled, show error
    if (!$userRegistrationEnabled && !$vendorRegistrationEnabled) {
        return redirect()->route('login')->with('error', 'Registration is currently disabled. Please contact support.');
    }
    
    return view('auth.register', compact('userRegistrationEnabled', 'vendorRegistrationEnabled'));
})->middleware('guest')->name('register');

// Handle registration
Route::post('/register', function (Request $request) {
    $isVendorRegistration = $request->has('register_as_vendor');
    
    // Check if the respective registration type is enabled
    if ($isVendorRegistration) {
        $vendorRegistrationEnabled = Setting::get('enable_vendor_registration', true);
        if (!$vendorRegistrationEnabled) {
            return back()->withErrors(['register_as_vendor' => 'Vendor registration is currently disabled.'])->withInput();
        }
    } else {
        $userRegistrationEnabled = Setting::get('enable_user_registration', true);
        if (!$userRegistrationEnabled) {
            return back()->withErrors(['email' => 'User registration is currently disabled.'])->withInput();
        }
    }
    
    $rules = [
        'name' => ['required', 'string', 'max:255'],
        'email' => ['required', 'string', 'email', 'max:255', 'unique:users'],
        'phone' => ['required', 'string', 'max:20'],
        'password' => ['required', 'confirmed', Password::defaults()],
        'terms' => ['accepted'],
        'captcha' => ['required', new \App\Rules\ValidCaptcha()],
    ];
    
    // Add vendor-specific validation
    if ($isVendorRegistration) {
        $rules['company_name'] = ['required', 'string', 'max:255'];
        $rules['company_description'] = ['nullable', 'string'];
        $rules['service_types'] = ['required', 'array', 'min:1'];
        $rules['service_types.*'] = ['in:property,car_rental'];
    }
    
    $validated = $request->validate($rules);

    $user = User::create([
        'name' => $validated['name'],
        'email' => $validated['email'],
        'phone' => $validated['phone'],
        'password' => Hash::make($validated['password']),
    ]);

    // If vendor registration, create vendor record and assign role
    if ($isVendorRegistration) {
        $vendor = Vendor::create([
            'user_id' => $user->id,
            'company_name' => $validated['company_name'],
            'company_description' => $validated['company_description'] ?? null,
            'commission_rate_at_signup' => Setting::get('global_commission_percentage', 15),
            'verified' => false,
            'service_types' => $validated['service_types'],
            'whatsapp_number' => $validated['phone'], // Use phone as WhatsApp number initially
        ]);
        
        $user->assignRole('vendor');
        
        Auth::login($user);
        
        // Send email verification notification
        $user->sendEmailVerificationNotification();
        
        return redirect()->route('verification.notice')->with('success', 'Vendor registration successful! Please verify your email address and complete your KYC verification.');
    }
    
    // Regular user registration
    $user->assignRole('user');
    Auth::login($user);
    
    // Send email verification notification
    $user->sendEmailVerificationNotification();

    return redirect()->route('verification.notice')->with('success', 'Registration successful! Please verify your email address.');
})->middleware('guest');

// Handle logout
Route::post('/logout', function (Request $request) {
    Auth::logout();
    $request->session()->invalidate();
    $request->session()->regenerateToken();
    return redirect('/');
})->middleware('auth')->name('logout');

// Password reset request form
Route::get('/forgot-password', function () {
    return view('auth.forgot-password');
})->middleware('guest')->name('password.request');

// Handle password reset request
Route::post('/forgot-password', function (Request $request) {
    $request->validate(['email' => 'required|email']);
    
    // Send password reset link
    $status = PasswordFacade::sendResetLink(
        $request->only('email')
    );
    
    return $status === PasswordFacade::RESET_LINK_SENT
        ? back()->with('status', __($status))
        : back()->withErrors(['email' => __($status)]);
})->middleware('guest')->name('password.email');

// Show password reset form
Route::get('/reset-password/{token}', function (string $token) {
    return view('auth.reset-password', ['token' => $token, 'email' => request('email')]);
})->middleware('guest')->name('password.reset');

// Handle password reset
Route::post('/reset-password', function (Request $request) {
    $request->validate([
        'token' => 'required',
        'email' => 'required|email',
        'password' => ['required', 'confirmed', Password::min(8)],
    ]);
    
    $status = PasswordFacade::reset(
        $request->only('email', 'password', 'password_confirmation', 'token'),
        function ($user, $password) {
            $user->forceFill([
                'password' => Hash::make($password)
            ])->setRememberToken(Str::random(60));
            
            $user->save();
            
            event(new PasswordReset($user));
        }
    );
    
    return $status === PasswordFacade::PASSWORD_RESET
        ? redirect()->route('login')->with('status', __($status))
        : back()->withErrors(['email' => [__($status)]]);
})->middleware('guest')->name('password.update');

// Email Verification Notice
Route::get('/email/verify', function () {
    return view('auth.verify');
})->middleware('auth')->name('verification.notice');

// Email Verification Handler
Route::get('/email/verify/{id}/{hash}', function (Request $request, $id, $hash) {
    $user = User::findOrFail($id);
    
    // Check if hash matches
    if (! hash_equals((string) $hash, sha1($user->getEmailForVerification()))) {
        return redirect()->route('verification.notice')
            ->with('error', 'Invalid verification link. Please request a new verification email.');
    }
    
    // Check if already verified
    if ($user->hasVerifiedEmail()) {
        return redirect()->route('home')->with('status', 'Email already verified!');
    }
    
    // Mark as verified
    if ($user->markEmailAsVerified()) {
        event(new \Illuminate\Auth\Events\Verified($user));
    }
    
    return redirect()->route('home')->with('status', 'Email verified successfully!');
})->middleware(['auth', 'signed'])->name('verification.verify');

// Resend Verification Email
Route::post('/email/verification-notification', function (Request $request) {
    if ($request->user()->hasVerifiedEmail()) {
        return redirect()->route('home');
    }
    
    $request->user()->sendEmailVerificationNotification();
    
    return back()->with('resent', true);
})->middleware(['auth', 'throttle:3,1'])->name('verification.send');

// Manual Email Verification (Development Only - Remove in Production)
Route::get('/email/verify-manual', function (Request $request) {
    if (!$request->user()) {
        return redirect()->route('login')->with('error', 'Please login first.');
    }
    
    if ($request->user()->hasVerifiedEmail()) {
        return redirect()->route('home')->with('status', 'Email already verified!');
    }
    
    if ($request->user()->markEmailAsVerified()) {
        event(new \Illuminate\Auth\Events\Verified($request->user()));
    }
    
    return redirect()->route('home')->with('status', 'Email verified successfully!');
})->middleware('auth')->name('verification.manual');

// Show terms page
Route::get('/terms', function () {
    return view('pages.terms');
})->name('terms');
