@extends('layouts.app')

@section('title', 'ShortStayNG - Book Hotels & Shortlets in Nigeria')

@push('styles')
<style>
    .hero-gradient {
        background: linear-gradient(135deg, #136D73 0%, #0f5a5f 100%);
    }
    .stat-card {
        transition: transform 0.3s ease;
    }
    .stat-card:hover {
        transform: translateY(-5px);
    }
</style>
@endpush

@section('content')

<!-- Featured Property Ad Modal -->
@php
    // Get a random active advertised property from all placement types
    $randomAd = \App\Models\VendorAd::with(['property.reviews', 'adPlan'])
        ->where('active', true)
        ->where('end_date', '>=', now())
        ->whereHas('property', function($query) {
            $query->where('status', 'active')
                  ->where('verified', true);
        })
        ->inRandomOrder()
        ->first();
@endphp

@if($randomAd && $randomAd->property)
<div id="ad-modal" class="hidden fixed inset-0 bg-black bg-opacity-75 z-50 flex items-center justify-center p-4">
    <div class="bg-white rounded-2xl max-w-2xl w-full relative overflow-hidden shadow-2xl">
        <!-- Close Button -->
        <button onclick="closeAdModal()" 
            class="absolute top-4 right-4 z-10 bg-white rounded-full w-10 h-10 flex items-center justify-center hover:bg-gray-100 shadow-lg">
            <i class="fas fa-times text-gray-600"></i>
        </button>

        <!-- Property Image -->
        <div class="relative">
            <img src="{{ isset($randomAd->property->photos[0]) ? asset('storage/' . $randomAd->property->photos[0]) : '/images/placeholder.jpg' }}" 
                alt="{{ $randomAd->property->title }}" 
                class="w-full h-64 object-cover">
            
            <!-- Dynamic Badge Based on Placement Type -->
            @if($randomAd->placement_type === 'featured')
                <div class="absolute top-4 left-4 bg-gradient-to-r from-blue-500 to-blue-600 text-white px-4 py-2 rounded-full font-bold shadow-lg">
                    <i class="fas fa-star mr-1"></i> FEATURED
                </div>
            @elseif($randomAd->placement_type === 'weekend_deals')
                <div class="absolute top-4 left-4 bg-gradient-to-r from-orange-500 to-orange-600 text-white px-4 py-2 rounded-full font-bold shadow-lg">
                    <i class="fas fa-calendar-week mr-1"></i> WEEKEND DEAL
                </div>
            @elseif($randomAd->placement_type === 'top_unique')
                <div class="absolute top-4 left-4 bg-gradient-to-r from-purple-500 to-purple-600 text-white px-4 py-2 rounded-full font-bold shadow-lg">
                    <i class="fas fa-gem mr-1"></i> TOP UNIQUE
                </div>
            @endif
        </div>

        <!-- Property Details -->
        <div class="p-6">
            <h2 class="text-2xl font-bold mb-2">{{ $randomAd->property->title }}</h2>
            <p class="text-gray-600 mb-4">
                <i class="fas fa-map-marker-alt mr-1"></i>
                {{ $randomAd->property->city }}, {{ $randomAd->property->state }}
            </p>

            <div class="grid grid-cols-3 gap-4 mb-4 text-sm">
                <div class="flex items-center">
                    <i class="fas fa-bed text-primary mr-2"></i>
                    <span>{{ $randomAd->property->bedrooms }} Bedrooms</span>
                </div>
                <div class="flex items-center">
                    <i class="fas fa-bath text-primary mr-2"></i>
                    <span>{{ $randomAd->property->bathrooms }} Bathrooms</span>
                </div>
                <div class="flex items-center">
                    <i class="fas fa-users text-primary mr-2"></i>
                    <span>{{ $randomAd->property->max_guests }} Guests</span>
                </div>
            </div>

            @if($avgRating = $randomAd->property->averageRating())
                <div class="flex items-center mb-4">
                    <div class="flex text-yellow-500">
                        @for($i = 1; $i <= 5; $i++)
                            <i class="fas fa-star{{ $i <= $avgRating ? '' : '-half-alt' }}"></i>
                        @endfor
                    </div>
                    <span class="ml-2 text-gray-600">{{ number_format($avgRating, 1) }} ({{ $randomAd->property->reviews->count() }} reviews)</span>
                </div>
            @endif

            <p class="text-gray-700 mb-4 line-clamp-2">{{ Str::limit($randomAd->property->description, 120) }}</p>

            <div class="flex items-center justify-between border-t pt-4">
                <div>
                    <p class="text-sm text-gray-600">Starting from</p>
                    <p class="text-3xl font-bold text-primary">₦{{ number_format($randomAd->property->price_per_night) }}</p>
                    <p class="text-sm text-gray-600">per night</p>
                </div>
                <a href="{{ route('properties.show', $randomAd->property) }}" 
                    class="bg-gradient-to-r from-primary to-primary-dark text-white px-8 py-3 rounded-lg hover:from-primary-dark hover:to-blue-800 font-semibold shadow-lg transform hover:scale-105 transition">
                    View Details <i class="fas fa-arrow-right ml-2"></i>
                </a>
            </div>
        </div>
    </div>
</div>

<script>
// Show property ad modal after 5 seconds on first page load
window.addEventListener('load', function() {
    const adModal = document.getElementById('ad-modal');
    const hasSeenAd = sessionStorage.getItem('adModalClosed');
    
    // Only show if user hasn't seen it this session
    if (!hasSeenAd && adModal) {
        setTimeout(function() {
            adModal.classList.remove('hidden');
        }, 5000); // Show after 5 seconds
    }
});

// Ad Modal Logic
function closeAdModal() {
    document.getElementById('ad-modal').classList.add('hidden');
    // Remember that user closed the modal (don't show again this session)
    sessionStorage.setItem('adModalClosed', 'true');
}

// Close modal when clicking outside
document.getElementById('ad-modal')?.addEventListener('click', function(e) {
    if (e.target === this) {
        closeAdModal();
    }
});

// Close modal on escape key
document.addEventListener('keydown', function(e) {
    if (e.key === 'Escape') {
        closeAdModal();
    }
});
</script>
@endif

<!-- Hero Section -->
<div class="hero-gradient text-white py-12 sm:py-24 mb-12 relative overflow-hidden">
    <div class="absolute inset-0 opacity-10">
        <div class="absolute transform rotate-45 -top-20 -right-20 w-96 h-96 bg-white rounded-full"></div>
        <div class="absolute transform -rotate-45 -bottom-20 -left-20 w-96 h-96 bg-white rounded-full"></div>
    </div>
    
    <div class="container mx-auto px-4 relative z-10">
        <div class="text-center max-w-4xl mx-auto">
            <h1 class="text-3xl sm:text-4xl md:text-5xl lg:text-6xl font-extrabold mb-4 sm:mb-6 leading-tight">
                Discover Your Perfect Stay<br>
                <span class="text-yellow-300">Across Nigeria</span>
            </h1>
            <p class="text-base sm:text-lg md:text-xl lg:text-2xl mb-6 sm:mb-10 opacity-90">
                Book verified hotels and shortlets with confidence. Over 1000+ properties nationwide.
            </p>
            
            <!-- Enhanced Search Form -->
            <form method="GET" action="{{ route('properties.index') }}" id="search-form" class="max-w-5xl mx-auto">
                <div class="bg-white rounded-xl sm:rounded-2xl shadow-2xl p-4 sm:p-6 md:p-8">
                    <div class="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-5 gap-3 sm:gap-4">
                        <div class="sm:col-span-2">
                            <label class="block text-gray-700 text-xs sm:text-sm font-semibold mb-2 text-left">Location</label>
                            <div class="relative">
                                <i class="fas fa-map-marker-alt absolute left-3 top-3 sm:top-4 text-gray-400"></i>
                                <input type="text" name="city" id="search-location" placeholder="City or State" 
                                    class="w-full border-2 border-gray-200 rounded-lg pl-10 pr-4 py-2 sm:py-3 text-sm sm:text-base text-secondary focus:border-primary focus:outline-none">
                            </div>
                        </div>
                        <div>
                            <label class="block text-gray-700 text-xs sm:text-sm font-semibold mb-2 text-left">Check-in</label>
                            <input type="date" name="check_in" id="search-checkin"
                                class="w-full border-2 border-gray-200 rounded-lg px-3 sm:px-4 py-2 sm:py-3 text-sm sm:text-base text-secondary focus:border-primary focus:outline-none">
                        </div>
                        <div>
                            <label class="block text-gray-700 text-xs sm:text-sm font-semibold mb-2 text-left">Check-out</label>
                            <input type="date" name="check_out" id="search-checkout"
                                class="w-full border-2 border-gray-200 rounded-lg px-3 sm:px-4 py-2 sm:py-3 text-sm sm:text-base text-secondary focus:border-primary focus:outline-none">
                        </div>
                        <div>
                            <label class="block text-gray-700 text-xs sm:text-sm font-semibold mb-2 text-left sm:invisible">&nbsp;</label>
                            <button type="submit" 
                                class="w-full bg-gradient-to-r from-primary to-primary text-white rounded-lg px-4 sm:px-6 py-2 sm:py-3 hover:from-primary-dark hover:to-primary-dark font-bold text-sm sm:text-base shadow-lg transform hover:scale-105 transition">
                                <i class="fas fa-search mr-2"></i> Search
                            </button>
                        </div>
                    </div>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Stats Section -->
<div class="container mx-auto px-4 -mt-8 sm:-mt-16 mb-12 sm:mb-16 relative z-20">
    <div class="grid grid-cols-2 md:grid-cols-4 gap-3 sm:gap-6">
        <div class="stat-card bg-white rounded-lg sm:rounded-xl shadow-lg p-4 sm:p-6 text-center">
            <div class="text-3xl sm:text-4xl md:text-5xl mb-2 sm:mb-3">
                <i class="fas fa-home text-primary"></i>
            </div>
            <div class="text-base sm:text-xl md:text-2xl font-bold text-primary mb-1 sm:mb-2">Premium Properties</div>
            <div class="text-xs sm:text-sm md:text-base text-gray-600">Verified listings nationwide</div>
        </div>
        <div class="stat-card bg-white rounded-lg sm:rounded-xl shadow-lg p-4 sm:p-6 text-center">
            <div class="text-3xl sm:text-4xl md:text-5xl mb-2 sm:mb-3">
                <i class="fas fa-users text-blue-600"></i>
            </div>
            <div class="text-base sm:text-xl md:text-2xl font-bold text-blue-600 mb-1 sm:mb-2">Trusted Community</div>
            <div class="text-xs sm:text-sm md:text-base text-gray-600">Thousands of happy guests</div>
        </div>
        <div class="stat-card bg-white rounded-lg sm:rounded-xl shadow-lg p-4 sm:p-6 text-center">
            <div class="text-3xl sm:text-4xl md:text-5xl mb-2 sm:mb-3">
                <i class="fas fa-shield-check text-green-600"></i>
            </div>
            <div class="text-base sm:text-xl md:text-2xl font-bold text-green-600 mb-1 sm:mb-2">100% Verified</div>
            <div class="text-xs sm:text-sm md:text-base text-gray-600">All vendors are verified</div>
        </div>
        <div class="stat-card bg-white rounded-lg sm:rounded-xl shadow-lg p-4 sm:p-6 text-center">
            <div class="text-3xl sm:text-4xl md:text-5xl mb-2 sm:mb-3">
                <i class="fas fa-star text-yellow-500"></i>
            </div>
            <div class="text-base sm:text-xl md:text-2xl font-bold text-yellow-500 mb-1 sm:mb-2">Top Rated</div>
            <div class="text-xs sm:text-sm md:text-base text-gray-600">Excellent customer reviews</div>
        </div>
    </div>
</div>

<!-- Recent Searches & Viewed Properties (Guest Activity) -->
<div id="guest-activity-section" class="container mx-auto px-4 mb-16" style="display: none;">
    <!-- Recent Searches -->
    <div id="recent-searches-container" class="mb-12" style="display: none;">
        <div class="flex items-center justify-between mb-6">
            <h2 class="text-3xl font-bold text-gray-800">
                <i class="fas fa-clock-rotate-left text-primary mr-2"></i>Your Recent Searches
            </h2>
            <button onclick="clearRecentSearches()" class="text-sm text-gray-600 hover:text-red-600 transition">
                <i class="fas fa-trash mr-1"></i>Clear All
            </button>
        </div>
        <div id="recent-searches-list" class="grid md:grid-cols-3 gap-4"></div>
    </div>

    <!-- Viewed Properties -->
    <div id="viewed-properties-container" class="mb-12" style="display: none;">
        <div class="flex items-center justify-between mb-6">
            <h2 class="text-3xl font-bold text-gray-800">
                <i class="fas fa-eye text-primary mr-2"></i>Still Interested in These Properties?
            </h2>
            <button onclick="clearViewedProperties()" class="text-sm text-gray-600 hover:text-red-600 transition">
                <i class="fas fa-trash mr-1"></i>Clear All
            </button>
        </div>
        <div id="viewed-properties-list" class="grid md:grid-cols-4 gap-6"></div>
    </div>
</div>

<!-- Why Choose Us -->
<div class="container mx-auto px-4 py-12 sm:py-16">
    <div class="text-center mb-8 sm:mb-12">
        <h2 class="text-2xl sm:text-3xl md:text-4xl font-bold mb-2">Why Choose ShortStayNG?</h2>
        <p class="text-sm sm:text-base text-gray-600">Your trusted partner for short stays across Nigeria</p>
    </div>
    
    <div class="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-4 gap-4 sm:gap-6 md:gap-8">
        <div class="bg-white rounded-xl p-6 sm:p-8 shadow-lg hover:shadow-2xl transition text-center">
            <div class="bg-gradient-to-br from-primary to-primary text-white w-16 h-16 sm:w-20 sm:h-20 rounded-2xl flex items-center justify-center mx-auto mb-3 sm:mb-4 shadow-lg">
                <i class="fas fa-shield-check text-2xl sm:text-3xl"></i>
            </div>
            <h3 class="font-bold text-lg sm:text-xl mb-2 sm:mb-3">100% Verified</h3>
            <p class="text-sm sm:text-base text-gray-600">All properties undergo strict verification for your safety and peace of mind</p>
        </div>

        <div class="bg-white rounded-xl p-6 sm:p-8 shadow-lg hover:shadow-2xl transition text-center">
            <div class="bg-gradient-to-br from-green-500 to-teal-600 text-white w-16 h-16 sm:w-20 sm:h-20 rounded-2xl flex items-center justify-center mx-auto mb-3 sm:mb-4 shadow-lg">
                <i class="fas fa-lock text-2xl sm:text-3xl"></i>
            </div>
            <h3 class="font-bold text-lg sm:text-xl mb-2 sm:mb-3">Secure Payments</h3>
            <p class="text-sm sm:text-base text-gray-600">Bank transfer, Paystack & Flutterwave - your money is always protected</p>
        </div>

        <div class="bg-white rounded-xl p-6 sm:p-8 shadow-lg hover:shadow-2xl transition text-center">
            <div class="bg-gradient-to-br from-orange-500 to-red-600 text-white w-16 h-16 sm:w-20 sm:h-20 rounded-2xl flex items-center justify-center mx-auto mb-3 sm:mb-4 shadow-lg">
                <i class="fas fa-headset text-2xl sm:text-3xl"></i>
            </div>
            <h3 class="font-bold text-lg sm:text-xl mb-2 sm:mb-3">24/7 Support</h3>
            <p class="text-sm sm:text-base text-gray-600">Round-the-clock customer service to assist you anytime, anywhere</p>
        </div>

        <div class="bg-white rounded-xl p-6 sm:p-8 shadow-lg hover:shadow-2xl transition text-center">
            <div class="bg-gradient-to-br from-yellow-500 to-orange-600 text-white w-16 h-16 sm:w-20 sm:h-20 rounded-2xl flex items-center justify-center mx-auto mb-3 sm:mb-4 shadow-lg">
                <i class="fas fa-tags text-2xl sm:text-3xl"></i>
            </div>
            <h3 class="font-bold text-lg sm:text-xl mb-2 sm:mb-3">Best Prices</h3>
            <p class="text-sm sm:text-base text-gray-600">Competitive rates with no hidden fees - what you see is what you pay</p>
        </div>
    </div>
</div>

<!-- Featured Properties -->
<div class="container mx-auto px-4 mb-12 sm:mb-16">
    <div class="flex flex-col sm:flex-row justify-between items-start sm:items-center mb-6 sm:mb-8 gap-3">
        <div>
            <h2 class="text-2xl sm:text-3xl md:text-4xl font-bold mb-1 sm:mb-2">Featured Properties</h2>
            <p class="text-sm sm:text-base text-gray-600">Handpicked premium stays for you</p>
        </div>
        <a href="{{ route('properties.index') }}" 
            class="text-primary hover:text-primary-dark font-semibold flex items-center text-sm sm:text-base">
            View All <i class="fas fa-arrow-right ml-2"></i>
        </a>
    </div>
    
    <div class="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4 sm:gap-6">
        @foreach(\App\Models\Property::active()->featured()->with('reviews')->limit(8)->get() as $property)
            <div class="bg-white rounded-xl shadow-lg overflow-hidden hover:shadow-2xl transition-all duration-300 transform hover:-translate-y-2">
                <div class="relative">
                    <img src="{{ isset($property->photos[0]) ? asset('storage/' . $property->photos[0]) : '/images/placeholder.jpg' }}" 
                        alt="{{ $property->title }}" 
                        class="w-full h-56 object-cover">
                    <div class="absolute top-3 left-3 bg-gradient-to-r from-yellow-400 to-orange-500 text-white px-3 py-1 rounded-full text-xs font-bold shadow-lg">
                        <i class="fas fa-star mr-1"></i> FEATURED
                    </div>
                    @if($property->allow_parties)
                        <div class="absolute top-3 right-3 bg-primary text-white px-3 py-1 rounded-full text-xs font-bold">
                            <i class="fas fa-glass-cheers mr-1"></i> Party OK
                        </div>
                    @endif
                </div>
                
                <div class="p-5">
                    <h3 class="font-bold text-lg mb-2 line-clamp-1">{{ $property->title }}</h3>
                    <p class="text-gray-600 text-sm mb-3 flex items-center">
                        <i class="fas fa-map-marker-alt mr-1 text-primary"></i>
                        {{ $property->city }}, {{ $property->state }}
                    </p>
                    
                    <div class="flex items-center gap-3 mb-3 text-xs text-gray-600">
                        <span><i class="fas fa-bed mr-1"></i>{{ $property->bedrooms }} Beds</span>
                        <span><i class="fas fa-bath mr-1"></i>{{ $property->bathrooms }} Baths</span>
                        <span><i class="fas fa-users mr-1"></i>{{ $property->max_guests }} Guests</span>
                    </div>
                    
                    @if($avgRating = $property->averageRating())
                        <div class="flex items-center mb-3">
                            <div class="flex text-yellow-500 text-sm">
                                @for($i = 1; $i <= 5; $i++)
                                    <i class="fas fa-star{{ $i <= $avgRating ? '' : ' text-gray-300' }}"></i>
                                @endfor
                            </div>
                            <span class="ml-2 text-sm text-gray-600">({{ $property->reviews->count() }})</span>
                        </div>
                    @endif
                    
                    <div class="border-t pt-3 flex justify-between items-center">
                        <div>
                            <span class="text-2xl font-bold text-primary">₦{{ number_format($property->price_per_night) }}</span>
                            <span class="text-sm text-gray-500">/night</span>
                        </div>
                        <a href="{{ route('properties.show', $property) }}" 
                            class="bg-primary text-white px-4 py-2 rounded-lg hover:bg-primary-dark transition font-semibold text-sm">
                            View
                        </a>
                    </div>
                </div>
            </div>
        @endforeach
    </div>
</div>




<!-- Popular Destinations -->
@php
    $popularDestinations = \App\Models\PopularDestination::active()->ordered()->limit(6)->get();
@endphp

@if($popularDestinations->count() > 0)
<div class="container mx-auto px-4 mb-12 sm:mb-16">
    <div class="text-center mb-8 sm:mb-12">
        <h2 class="text-2xl sm:text-3xl md:text-4xl font-bold mb-2">Explore Popular Destinations</h2>
        <p class="text-sm sm:text-base text-gray-600">Discover amazing places across Nigeria</p>
    </div>
    
    <div class="grid grid-cols-2 sm:grid-cols-3 md:grid-cols-4 lg:grid-cols-6 gap-3 sm:gap-4">
        @foreach($popularDestinations as $destination)
            <a href="{{ $destination->url }}" 
                class="group relative overflow-hidden rounded-lg sm:rounded-xl shadow-lg hover:shadow-2xl transition-all duration-300 h-32 sm:h-40 md:h-48">
                <!-- Background Image -->
                <div class="absolute inset-0">
                    <img src="{{ asset('storage/' . $destination->image) }}" 
                        alt="{{ $destination->name }}" 
                        class="w-full h-full object-cover group-hover:scale-110 transition-transform duration-500 brightness-90">
                </div>
                <!-- Content -->
                <div class="absolute inset-0 flex flex-col justify-end p-3 sm:p-4">
                    <div class="transform group-hover:-translate-y-1 transition-transform duration-300">
                        <div class="flex items-center mb-1">
                            <i class="fas fa-{{ $destination->type === 'state' ? 'flag' : 'city' }} mr-1 text-white text-xs"></i>
                            <span class="text-xs font-semibold text-white uppercase tracking-wide">
                                {{ ucfirst($destination->type) }}
                            </span>
                        </div>
                        <h3 class="text-sm sm:text-base md:text-lg font-bold mb-1 text-white drop-shadow-lg">{{ $destination->name }}</h3>
                        @if($destination->type === 'city' && $destination->state)
                            <p class="text-xs text-white drop-shadow-lg mb-1">{{ $destination->state }}</p>
                        @endif
                        <p class="text-white text-xs flex items-center drop-shadow-lg">
                            <i class="fas fa-home mr-1"></i>
                            {{ $destination->property_count }} Properties
                        </p>
                    </div>
                </div>
            </a>
        @endforeach
    </div>
</div>
@endif
<!-- Top Unique Properties -->
@php
    // Get properties with ACTIVE top_unique ads
    $topUniquePropertyIds = \App\Models\VendorAd::where('active', true)
        ->where('placement_type', 'top_unique')
        ->where('end_date', '>=', now())
        ->pluck('property_id');
    
    $topUniqueProperties = \App\Models\Property::active()
        ->whereIn('id', $topUniquePropertyIds)
        ->with('reviews')
        ->limit(8)
        ->get();
@endphp

@if($topUniqueProperties->isNotEmpty())
<div class="container mx-auto px-4 mb-12 sm:mb-16">
    <div class="flex flex-col sm:flex-row justify-between items-start sm:items-center mb-6 sm:mb-8 gap-3">
        <div>
            <h2 class="text-2xl sm:text-3xl md:text-4xl font-bold mb-1 sm:mb-2">Stay at Our Top Unique Properties</h2>
            <p class="text-sm sm:text-base text-gray-600">Curated collection of exceptional stays</p>
        </div>
        <a href="{{ route('properties.index') }}" 
            class="text-purple-600 hover:text-purple-700 font-semibold flex items-center text-sm sm:text-base">
            View All <i class="fas fa-arrow-right ml-2"></i>
        </a>
    </div>
    
    <div class="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4 sm:gap-6">
        @foreach($topUniqueProperties as $property)
            <div class="bg-white rounded-xl shadow-lg overflow-hidden hover:shadow-2xl transition-all duration-300 transform hover:-translate-y-2">
                <div class="relative">
                    <img src="{{ isset($property->photos[0]) ? asset('storage/' . $property->photos[0]) : '/images/placeholder.jpg' }}" 
                        alt="{{ $property->title }}" 
                        class="w-full h-56 object-cover">
                    <div class="absolute top-3 left-3 bg-gradient-to-r from-purple-500 to-purple-600 text-white px-3 py-1 rounded-full text-xs font-bold shadow-lg">
                        <i class="fas fa-gem mr-1"></i> TOP UNIQUE
                    </div>
                    @if($property->allow_parties)
                        <div class="absolute top-3 right-3 bg-primary text-white px-3 py-1 rounded-full text-xs font-bold">
                            <i class="fas fa-glass-cheers mr-1"></i> Party OK
                        </div>
                    @endif
                </div>
                
                <div class="p-5">
                    <h3 class="font-bold text-lg mb-2 line-clamp-1">{{ $property->title }}</h3>
                    <p class="text-gray-600 text-sm mb-3 flex items-center">
                        <i class="fas fa-map-marker-alt mr-1 text-primary"></i>
                        {{ $property->city }}, {{ $property->state }}
                    </p>
                    
                    <div class="flex items-center gap-3 mb-3 text-xs text-gray-600">
                        <span><i class="fas fa-bed mr-1"></i>{{ $property->bedrooms }} Beds</span>
                        <span><i class="fas fa-bath mr-1"></i>{{ $property->bathrooms }} Baths</span>
                        <span><i class="fas fa-users mr-1"></i>{{ $property->max_guests }} Guests</span>
                    </div>
                    
                    @if($avgRating = $property->averageRating())
                        <div class="flex items-center mb-3">
                            <div class="flex text-yellow-500 text-sm">
                                @for($i = 1; $i <= 5; $i++)
                                    <i class="fas fa-star{{ $i <= $avgRating ? '' : ' text-gray-300' }}"></i>
                                @endfor
                            </div>
                            <span class="ml-2 text-sm text-gray-600">({{ $property->reviews->count() }})</span>
                        </div>
                    @endif
                    
                    <div class="border-t pt-3 flex justify-between items-center">
                        <div>
                            <span class="text-2xl font-bold text-primary">₦{{ number_format($property->price_per_night) }}</span>
                            <span class="text-sm text-gray-500">/night</span>
                        </div>
                        <a href="{{ route('properties.show', $property) }}" 
                            class="bg-purple-500 text-white px-4 py-2 rounded-lg hover:bg-purple-600 transition font-semibold text-sm">
                            View
                        </a>
                    </div>
                </div>
            </div>
        @endforeach
    </div>
</div>
@endif

<!-- Special Offers / Discount Banner -->
@php
    $homepageDeal = \App\Models\DiscountDeal::where('is_active', true)
        ->where('show_on_homepage', true)
        ->where(function($q) {
            $q->whereNull('valid_from')->orWhere('valid_from', '<=', now());
        })
        ->where(function($q) {
            $q->whereNull('valid_until')->orWhere('valid_until', '>=', now());
        })
        ->first();
@endphp

@if($homepageDeal)
<div class="container mx-auto px-4 py-8 sm:py-12">
    <div class="bg-gradient-to-r from-{{ $homepageDeal->banner_bg_from }} to-{{ $homepageDeal->banner_bg_to }} rounded-xl sm:rounded-2xl p-6 sm:p-8 text-white relative overflow-hidden">
        <div class="absolute top-0 right-0 opacity-10">
            <i class="fas fa-{{ $homepageDeal->banner_icon }} text-6xl sm:text-7xl md:text-9xl"></i>
        </div>
        <div class="relative z-10">
            <div class="flex flex-col sm:flex-row items-start sm:items-center justify-between gap-4">
                <div class="flex-1">
                    <h2 class="text-2xl sm:text-3xl md:text-4xl font-bold mb-2">{{ $homepageDeal->banner_title ?? $homepageDeal->name }}</h2>
                    <p class="text-base sm:text-lg md:text-xl mb-3 sm:mb-4">{{ $homepageDeal->banner_description ?? $homepageDeal->description }}</p>
                    <p class="text-xs sm:text-sm opacity-90">
                        Use code: <span class="bg-white text-{{ $homepageDeal->banner_bg_from }} px-2 sm:px-3 py-1 rounded font-bold text-sm sm:text-base">{{ $homepageDeal->code }}</span>
                        <span class="ml-2">- 
                            @if($homepageDeal->discount_type === 'percentage')
                                {{ $homepageDeal->discount_value }}% OFF
                            @else
                                Save ₦{{ number_format($homepageDeal->discount_value) }}
                            @endif
                        </span>
                    </p>
                </div>
                <a href="{{ $homepageDeal->banner_button_link }}" 
                   class="w-full sm:w-auto bg-white text-{{ $homepageDeal->banner_bg_from }} px-6 sm:px-8 py-3 sm:py-4 rounded-lg sm:rounded-xl font-bold text-base sm:text-lg hover:bg-gray-100 transition text-center whitespace-nowrap">
                    {{ $homepageDeal->banner_button_text }}
                </a>
            </div>
        </div>
    </div>
</div>
@endif

<!-- Browse by Property Type -->
@php
    $houseTypeImages = \App\Models\HouseTypeImage::active()->get();
@endphp

@if($houseTypeImages->count() > 0)
<div class="container mx-auto px-4 py-12 sm:py-16">
    <div class="text-center mb-8 sm:mb-12">
        <h2 class="text-2xl sm:text-3xl md:text-4xl font-bold mb-2">Browse by Property Type</h2>
        <p class="text-sm sm:text-base text-gray-600">Find your perfect accommodation style</p>
    </div>
    
    <div class="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4 sm:gap-6">
        @foreach($houseTypeImages as $houseType)
            <a href="{{ route('properties.index', ['house_type' => $houseType->house_type]) }}" 
                class="group relative overflow-hidden rounded-xl sm:rounded-2xl shadow-lg hover:shadow-2xl transition-all duration-300 h-48 sm:h-56 md:h-64 transform hover:-translate-y-2">
                <!-- Background Image -->
                <div class="absolute inset-0">
                    <img src="{{ asset('storage/' . $houseType->image) }}" 
                        alt="{{ $houseType->house_type }}" 
                        class="w-full h-full object-cover group-hover:scale-110 transition-transform duration-500 brightness-95">
                </div>
                <!-- Light Overlay -->
                <div class="absolute inset-0 bg-gradient-to-t from-black/70 via-black/20 to-transparent group-hover:from-black/80 transition"></div>
                <!-- Content -->
                <div class="absolute inset-0 flex flex-col justify-end p-4 sm:p-6 text-white">
                    <div class="transform group-hover:-translate-y-2 transition-transform duration-300">
                        <h3 class="text-lg sm:text-xl md:text-2xl font-bold mb-1 sm:mb-2 drop-shadow-lg">{{ $houseType->house_type }}</h3>
                        @if($houseType->description)
                            <p class="text-xs sm:text-sm text-white/90 mb-2 sm:mb-3 drop-shadow-lg line-clamp-2">{{ $houseType->description }}</p>
                        @endif
                        <p class="text-white/90 flex items-center drop-shadow-lg text-xs sm:text-sm">
                            <i class="fas fa-home mr-2"></i>
                            <span class="font-semibold">{{ number_format($houseType->property_count) }}</span>
                            <span class="ml-1">{{ $houseType->property_count == 1 ? 'Property' : 'Properties' }}</span>
                        </p>
                    </div>
                </div>
            </a>
        @endforeach
    </div>
</div>
@endif

<!-- Weekend Deals -->
@php
    // Get properties with ACTIVE weekend_deals ads
    $weekendDealsPropertyIds = \App\Models\VendorAd::where('active', true)
        ->where('placement_type', 'weekend_deals')
        ->where('end_date', '>=', now())
        ->pluck('property_id');
    
    $weekendDealsProperties = \App\Models\Property::active()
        ->whereIn('id', $weekendDealsPropertyIds)
        ->with('reviews')
        ->limit(8)
        ->get();
@endphp

@if($weekendDealsProperties->isNotEmpty())
<div class="container mx-auto px-4 mb-12 sm:mb-16">
    <div class="flex flex-col sm:flex-row justify-between items-start sm:items-center mb-6 sm:mb-8 gap-3">
        <div>
            <h2 class="text-2xl sm:text-3xl md:text-4xl font-bold mb-1 sm:mb-2">Deals for the Weekend</h2>
            <p class="text-sm sm:text-base text-gray-600">Special weekend offers on amazing properties</p>
        </div>
        <a href="{{ route('properties.index') }}" 
            class="text-orange-600 hover:text-orange-700 font-semibold flex items-center text-sm sm:text-base">
            View All <i class="fas fa-arrow-right ml-2"></i>
        </a>
    </div>
    
    <div class="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4 sm:gap-6">
        @foreach($weekendDealsProperties as $property)
            <div class="bg-white rounded-xl shadow-lg overflow-hidden hover:shadow-2xl transition-all duration-300 transform hover:-translate-y-2">
                <div class="relative">
                    <img src="{{ isset($property->photos[0]) ? asset('storage/' . $property->photos[0]) : '/images/placeholder.jpg' }}" 
                        alt="{{ $property->title }}" 
                        class="w-full h-56 object-cover">
                    <div class="absolute top-3 left-3 bg-gradient-to-r from-orange-500 to-orange-600 text-white px-3 py-1 rounded-full text-xs font-bold shadow-lg">
                        <i class="fas fa-calendar-week mr-1"></i> WEEKEND DEAL
                    </div>
                    @if($property->allow_parties)
                        <div class="absolute top-3 right-3 bg-primary text-white px-3 py-1 rounded-full text-xs font-bold">
                            <i class="fas fa-glass-cheers mr-1"></i> Party OK
                        </div>
                    @endif
                </div>
                
                <div class="p-5">
                    <h3 class="font-bold text-lg mb-2 line-clamp-1">{{ $property->title }}</h3>
                    <p class="text-gray-600 text-sm mb-3 flex items-center">
                        <i class="fas fa-map-marker-alt mr-1 text-primary"></i>
                        {{ $property->city }}, {{ $property->state }}
                    </p>
                    
                    <div class="flex items-center gap-3 mb-3 text-xs text-gray-600">
                        <span><i class="fas fa-bed mr-1"></i>{{ $property->bedrooms }} Beds</span>
                        <span><i class="fas fa-bath mr-1"></i>{{ $property->bathrooms }} Baths</span>
                        <span><i class="fas fa-users mr-1"></i>{{ $property->max_guests }} Guests</span>
                    </div>
                    
                    @if($avgRating = $property->averageRating())
                        <div class="flex items-center mb-3">
                            <div class="flex text-yellow-500 text-sm">
                                @for($i = 1; $i <= 5; $i++)
                                    <i class="fas fa-star{{ $i <= $avgRating ? '' : ' text-gray-300' }}"></i>
                                @endfor
                            </div>
                            <span class="ml-2 text-sm text-gray-600">({{ $property->reviews->count() }})</span>
                        </div>
                    @endif
                    
                    <div class="border-t pt-3 flex justify-between items-center">
                        <div>
                            <span class="text-2xl font-bold text-primary">₦{{ number_format($property->price_per_night) }}</span>
                            <span class="text-sm text-gray-500">/night</span>
                        </div>
                        <a href="{{ route('properties.show', $property) }}" 
                            class="bg-orange-500 text-white px-4 py-2 rounded-lg hover:bg-orange-600 transition font-semibold text-sm">
                            View
                        </a>
                    </div>
                </div>
            </div>
        @endforeach
    </div>
</div>
@endif


<script>
// Set minimum date for check-in to today
document.addEventListener('DOMContentLoaded', function() {
    const today = new Date().toISOString().split('T')[0];
    const checkinInput = document.getElementById('search-checkin');
    const checkoutInput = document.getElementById('search-checkout');
    
    if (checkinInput) {
        checkinInput.setAttribute('min', today);
        checkinInput.addEventListener('change', function() {
            const checkinDate = new Date(this.value);
            checkinDate.setDate(checkinDate.getDate() + 1);
            checkoutInput.setAttribute('min', checkinDate.toISOString().split('T')[0]);
            if (checkoutInput.value && new Date(checkoutInput.value) <= new Date(this.value)) {
                checkoutInput.value = checkinDate.toISOString().split('T')[0];
            }
        });
    }
});

// Track search form submission
document.getElementById('search-form')?.addEventListener('submit', function(e) {
    const location = document.getElementById('search-location').value;
    const checkIn = document.getElementById('search-checkin').value;
    const checkOut = document.getElementById('search-checkout').value;
    
    console.log('Tracking search:', { location, checkIn, checkOut });
    
    // Track search (don't prevent form submission)
    fetch('/guest/track-search', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
        },
        body: JSON.stringify({
            location: location,
            check_in: checkIn,
            check_out: checkOut,
            guests: 1
        })
    })
    .then(response => response.json())
    .then(data => console.log('Search tracked successfully:', data))
    .catch(error => console.error('Error tracking search:', error));
});

// Guest Activity Tracking
document.addEventListener('DOMContentLoaded', function() {
    loadGuestActivity();
});

async function loadGuestActivity() {
    try {
        // Load recent searches
        const searchesResponse = await fetch('/guest/recent-searches');
        const searches = await searchesResponse.json();
        
        console.log('Recent searches loaded:', searches.length, searches);
        
        if (searches.length > 0) {
            displayRecentSearches(searches);
        }
        
        // Load viewed properties
        const propertiesResponse = await fetch('/guest/viewed-properties');
        const properties = await propertiesResponse.json();
        
        console.log('Viewed properties loaded:', properties.length, properties);
        
        if (properties.length > 0) {
            displayViewedProperties(properties);
        }
        
        // Show section if there's any activity
        if (searches.length > 0 || properties.length > 0) {
            document.getElementById('guest-activity-section').style.display = 'block';
            console.log('Guest activity section displayed');
        } else {
            console.log('No guest activity to display yet. Perform a search or view a property first.');
        }
    } catch (error) {
        console.error('Error loading guest activity:', error);
    }
}

function displayRecentSearches(searches) {
    const container = document.getElementById('recent-searches-list');
    const searchesContainer = document.getElementById('recent-searches-container');
    
    container.innerHTML = searches.map(search => {
        const checkIn = search.check_in ? new Date(search.check_in).toLocaleDateString('en-US', { month: 'short', day: 'numeric' }) : '';
        const checkOut = search.check_out ? new Date(search.check_out).toLocaleDateString('en-US', { month: 'short', day: 'numeric' }) : '';
        const dates = checkIn && checkOut ? `${checkIn}–${checkOut}` : 'Dates not set';
        const guests = search.guests || 1;
        
        const searchUrl = `/properties?location=${encodeURIComponent(search.location || '')}&check_in=${search.check_in || ''}&check_out=${search.check_out || ''}&guests=${guests}`;
        
        return `
            <a href="${searchUrl}" class="block bg-white rounded-xl shadow-lg hover:shadow-2xl transition-all duration-300 p-6 border-2 border-transparent hover:border-primary">
                <div class="flex items-start justify-between mb-3">
                    <div class="flex-1">
                        <h3 class="text-xl font-bold text-gray-800 mb-1">${search.location || 'All Locations'}</h3>
                        <p class="text-sm text-gray-600">
                            <i class="fas fa-calendar mr-1 text-primary"></i>${dates}
                        </p>
                    </div>
                    <div class="bg-primary/10 text-primary px-3 py-1 rounded-full text-sm font-semibold">
                        <i class="fas fa-user mr-1"></i>${guests}
                    </div>
                </div>
                <div class="flex items-center text-primary font-semibold text-sm">
                    <i class="fas fa-search mr-2"></i>Search Again
                </div>
            </a>
        `;
    }).join('');
    
    searchesContainer.style.display = 'block';
}

function displayViewedProperties(properties) {
    const container = document.getElementById('viewed-properties-list');
    const propertiesContainer = document.getElementById('viewed-properties-container');
    
    container.innerHTML = properties.slice(0, 4).map(property => {
        const image = property.photos && property.photos[0] ? `/storage/${property.photos[0]}` : '/images/placeholder.jpg';
        const rating = property.average_rating || 0;
        const reviewCount = property.reviews_count || 0;
        
        return `
            <a href="/properties/${property.slug}" class="block bg-white rounded-xl shadow-lg hover:shadow-2xl transition-all duration-300 overflow-hidden group">
                <div class="relative h-48">
                    <img src="${image}" alt="${property.title}" class="w-full h-full object-cover group-hover:scale-110 transition-transform duration-500">
                    <div class="absolute top-3 right-3 bg-white px-3 py-1 rounded-full shadow-lg">
                        <i class="fas fa-star text-yellow-500"></i>
                        <span class="font-bold text-gray-800 ml-1">${rating.toFixed(1)}</span>
                    </div>
                </div>
                <div class="p-4">
                    <h3 class="font-bold text-lg text-gray-800 mb-2 line-clamp-2">${property.title}</h3>
                    <p class="text-sm text-gray-600 mb-2">
                        <i class="fas fa-map-marker-alt text-primary mr-1"></i>${property.city}, ${property.state}
                    </p>
                    <div class="flex items-center justify-between">
                        <span class="text-primary font-bold text-lg">₦${Number(property.price_per_night).toLocaleString()}</span>
                        <span class="text-xs text-gray-500">${reviewCount} reviews</span>
                    </div>
                </div>
            </a>
        `;
    }).join('');
    
    propertiesContainer.style.display = 'block';
}

async function clearRecentSearches() {
    if (!confirm('Clear all recent searches?')) return;
    
    try {
        await fetch('/guest/clear-activity', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
            },
            body: JSON.stringify({ type: 'searches' })
        });
        
        document.getElementById('recent-searches-container').style.display = 'none';
        checkActivitySection();
    } catch (error) {
        console.error('Error clearing searches:', error);
    }
}

async function clearViewedProperties() {
    if (!confirm('Clear all viewed properties?')) return;
    
    try {
        await fetch('/guest/clear-activity', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
            },
            body: JSON.stringify({ type: 'properties' })
        });
        
        document.getElementById('viewed-properties-container').style.display = 'none';
        checkActivitySection();
    } catch (error) {
        console.error('Error clearing properties:', error);
    }
}

function checkActivitySection() {
    const searchesVisible = document.getElementById('recent-searches-container').style.display !== 'none';
    const propertiesVisible = document.getElementById('viewed-properties-container').style.display !== 'none';
    
    if (!searchesVisible && !propertiesVisible) {
        document.getElementById('guest-activity-section').style.display = 'none';
    }
}
</script>
@endsection