@extends('layouts.app')

@section('title', 'Book ' . $car->full_name)

@section('content')
<div class="min-h-screen bg-gray-50 py-4 sm:py-8">
    <div class="container mx-auto px-3 sm:px-4 max-w-6xl">
        <!-- Header -->
        <div class="mb-4 sm:mb-8">
            <div class="bg-gradient-to-r from-green-600 to-teal-600 text-white rounded-lg sm:rounded-xl shadow-lg p-4 sm:p-6">
                <h1 class="text-xl sm:text-2xl md:text-3xl font-bold mb-1 sm:mb-2">
                    <i class="fas fa-calendar-check mr-2 sm:mr-3"></i>Book Your Car
                </h1>
                <p class="text-green-100 text-sm sm:text-base">{{ $car->full_name }}</p>
            </div>
        </div>

        <form method="POST" action="{{ route('cars.booking.store', $car) }}" enctype="multipart/form-data">
            @csrf

            <div class="grid lg:grid-cols-3 gap-4 sm:gap-6 lg:gap-8">
                <!-- Booking Form -->
                <div class="lg:col-span-2 space-y-4 sm:space-y-6">
                    <!-- Rental Dates -->
                    <div class="bg-white rounded-lg sm:rounded-xl shadow-lg p-4 sm:p-6">
                        <h2 class="text-lg sm:text-xl font-bold text-gray-800 mb-4 sm:mb-6 flex items-center">
                            <i class="fas fa-calendar text-green-600 mr-2 sm:mr-3"></i>Rental Period
                        </h2>

                        <div class="grid md:grid-cols-2 gap-4 sm:gap-6">
                            <!-- Pickup Date & Time -->
                            <div>
                                <label class="block text-sm font-semibold text-gray-700 mb-2">Pickup Date *</label>
                                <input type="date" name="pickup_date" value="{{ old('pickup_date') }}" required
                                    min="{{ date('Y-m-d', strtotime('+1 day')) }}"
                                    class="w-full border-2 border-gray-300 rounded-lg px-3 sm:px-4 py-2 sm:py-3 text-sm sm:text-base focus:border-green-600 focus:outline-none @error('pickup_date') border-red-500 @enderror">
                                @error('pickup_date')
                                    <p class="text-red-500 text-xs sm:text-sm mt-1">{{ $message }}</p>
                                @enderror
                            </div>

                            <div>
                                <label class="block text-sm font-semibold text-gray-700 mb-2">Pickup Time *</label>
                                <input type="time" name="pickup_time" value="{{ old('pickup_time', '09:00') }}" required
                                    class="w-full border-2 border-gray-300 rounded-lg px-3 sm:px-4 py-2 sm:py-3 text-sm sm:text-base focus:border-green-600 focus:outline-none @error('pickup_time') border-red-500 @enderror">
                                @error('pickup_time')
                                    <p class="text-red-500 text-xs sm:text-sm mt-1">{{ $message }}</p>
                                @enderror
                            </div>

                            <!-- Return Date & Time -->
                            <div>
                                <label class="block text-sm font-semibold text-gray-700 mb-2">Return Date *</label>
                                <input type="date" name="return_date" value="{{ old('return_date') }}" required
                                    min="{{ date('Y-m-d', strtotime('+2 days')) }}"
                                    class="w-full border-2 border-gray-300 rounded-lg px-3 sm:px-4 py-2 sm:py-3 text-sm sm:text-base focus:border-green-600 focus:outline-none @error('return_date') border-red-500 @enderror">
                                @error('return_date')
                                    <p class="text-red-500 text-xs sm:text-sm mt-1">{{ $message }}</p>
                                @enderror
                            </div>

                            <div>
                                <label class="block text-sm font-semibold text-gray-700 mb-2">Return Time *</label>
                                <input type="time" name="return_time" value="{{ old('return_time', '09:00') }}" required
                                    class="w-full border-2 border-gray-300 rounded-lg px-3 sm:px-4 py-2 sm:py-3 text-sm sm:text-base focus:border-green-600 focus:outline-none @error('return_time') border-red-500 @enderror">
                                @error('return_time')
                                    <p class="text-red-500 text-xs sm:text-sm mt-1">{{ $message }}</p>
                                @enderror
                            </div>
                        </div>

                        <div class="mt-3 sm:mt-4 p-3 sm:p-4 bg-blue-50 rounded-lg">
                            <p class="text-xs sm:text-sm text-blue-800">
                                <i class="fas fa-info-circle mr-2"></i>
                                @if($car->maximum_rental_days)
                                    Rental period: {{ $car->minimum_rental_days }} - {{ $car->maximum_rental_days }} days
                                @else
                                    Minimum rental period: {{ $car->minimum_rental_days }} {{ Str::plural('day', $car->minimum_rental_days) }}
                                @endif
                            </p>
                        </div>

                        <div id="rental-days-warning" class="mt-3 sm:mt-4 p-3 sm:p-4 bg-red-50 border-l-4 border-red-500 rounded hidden">
                            <p class="text-xs sm:text-sm text-red-800 font-semibold">
                                <i class="fas fa-exclamation-triangle mr-2"></i>
                                <span id="warning-message"></span>
                            </p>
                        </div>
                    </div>

                    <!-- Driver Information -->
                    <div class="bg-white rounded-lg sm:rounded-xl shadow-lg p-4 sm:p-6">
                        <h2 class="text-lg sm:text-xl font-bold text-gray-800 mb-4 sm:mb-6 flex items-center">
                            <i class="fas fa-user text-green-600 mr-2 sm:mr-3"></i>Driver Information
                        </h2>

                        <div class="grid md:grid-cols-2 gap-4 sm:gap-6">
                            <div>
                                <label class="block text-sm font-semibold text-gray-700 mb-2">Full Name *</label>
                                <input type="text" name="driver_name" value="{{ old('driver_name', auth()->user()->name) }}" required
                                    class="w-full border-2 border-gray-300 rounded-lg px-3 sm:px-4 py-2 sm:py-3 text-sm sm:text-base focus:border-green-600 focus:outline-none @error('driver_name') border-red-500 @enderror">
                                @error('driver_name')
                                    <p class="text-red-500 text-xs sm:text-sm mt-1">{{ $message }}</p>
                                @enderror
                            </div>

                            <div>
                                <label class="block text-sm font-semibold text-gray-700 mb-2">Phone Number *</label>
                                <input type="tel" name="driver_phone" value="{{ old('driver_phone', auth()->user()->phone) }}" required
                                    class="w-full border-2 border-gray-300 rounded-lg px-3 sm:px-4 py-2 sm:py-3 text-sm sm:text-base focus:border-green-600 focus:outline-none @error('driver_phone') border-red-500 @enderror">
                                @error('driver_phone')
                                    <p class="text-red-500 text-xs sm:text-sm mt-1">{{ $message }}</p>
                                @enderror
                            </div>

                            <div>
                                <label class="block text-sm font-semibold text-gray-700 mb-2">Email *</label>
                                <input type="email" name="driver_email" value="{{ old('driver_email', auth()->user()->email) }}" required
                                    class="w-full border-2 border-gray-300 rounded-lg px-3 sm:px-4 py-2 sm:py-3 text-sm sm:text-base focus:border-green-600 focus:outline-none @error('driver_email') border-red-500 @enderror">
                                @error('driver_email')
                                    <p class="text-red-500 text-xs sm:text-sm mt-1">{{ $message }}</p>
                                @enderror
                            </div>

                            <div>
                                <label class="block text-sm font-semibold text-gray-700 mb-2">Age *</label>
                                <input type="number" name="driver_age" value="{{ old('driver_age') }}" required
                                    min="{{ $car->minimum_driver_age }}"
                                    class="w-full border-2 border-gray-300 rounded-lg px-3 sm:px-4 py-2 sm:py-3 text-sm sm:text-base focus:border-green-600 focus:outline-none @error('driver_age') border-red-500 @enderror">
                                @error('driver_age')
                                    <p class="text-red-500 text-xs sm:text-sm mt-1">{{ $message }}</p>
                                @enderror
                                <p class="text-xs text-gray-500 mt-1">Minimum age: {{ $car->minimum_driver_age }} years</p>
                            </div>

                            <div>
                                <label class="block text-sm font-semibold text-gray-700 mb-2">License Number *</label>
                                <input type="text" name="driver_license_number" value="{{ old('driver_license_number') }}" required
                                    class="w-full border-2 border-gray-300 rounded-lg px-3 sm:px-4 py-2 sm:py-3 text-sm sm:text-base focus:border-green-600 focus:outline-none @error('driver_license_number') border-red-500 @enderror">
                                @error('driver_license_number')
                                    <p class="text-red-500 text-xs sm:text-sm mt-1">{{ $message }}</p>
                                @enderror
                            </div>

                            <div>
                                <label class="block text-sm font-semibold text-gray-700 mb-2">License Expiry Date *</label>
                                <input type="date" name="driver_license_expiry" value="{{ old('driver_license_expiry') }}" required
                                    min="{{ date('Y-m-d') }}"
                                    class="w-full border-2 border-gray-300 rounded-lg px-3 sm:px-4 py-2 sm:py-3 text-sm sm:text-base focus:border-green-600 focus:outline-none @error('driver_license_expiry') border-red-500 @enderror">
                                @error('driver_license_expiry')
                                    <p class="text-red-500 text-xs sm:text-sm mt-1">{{ $message }}</p>
                                @enderror
                            </div>

                            <div>
                                <label class="block text-sm font-semibold text-gray-700 mb-2">Driving Experience (Years) *</label>
                                <input type="number" name="driver_experience_years" value="{{ old('driver_experience_years') }}" required
                                    min="{{ $car->minimum_license_years }}"
                                    class="w-full border-2 border-gray-300 rounded-lg px-3 sm:px-4 py-2 sm:py-3 text-sm sm:text-base focus:border-green-600 focus:outline-none @error('driver_experience_years') border-red-500 @enderror">
                                @error('driver_experience_years')
                                    <p class="text-red-500 text-xs sm:text-sm mt-1">{{ $message }}</p>
                                @enderror
                                <p class="text-xs text-gray-500 mt-1">Minimum: {{ $car->minimum_license_years }} years</p>
                            </div>

                            <div>
                                <label class="block text-sm font-semibold text-gray-700 mb-2">Driver's License Photo *</label>
                                <input type="file" name="driver_license_photo" accept=".pdf,.jpg,.jpeg,.png" required
                                    class="w-full border-2 border-gray-300 rounded-lg px-3 sm:px-4 py-2 sm:py-3 text-sm focus:border-green-600 focus:outline-none @error('driver_license_photo') border-red-500 @enderror">
                                @error('driver_license_photo')
                                    <p class="text-red-500 text-xs sm:text-sm mt-1">{{ $message }}</p>
                                @enderror
                                <p class="text-xs text-gray-500 mt-1">PDF, JPG, or PNG (Max 5MB)</p>
                            </div>
                        </div>
                    </div>

                    <!-- Additional Options -->
                    <div class="bg-white rounded-lg sm:rounded-xl shadow-lg p-4 sm:p-6">
                        <h2 class="text-lg sm:text-xl font-bold text-gray-800 mb-4 sm:mb-6 flex items-center">
                            <i class="fas fa-plus-circle text-green-600 mr-2 sm:mr-3"></i>Additional Options
                        </h2>

                        <div class="space-y-3 sm:space-y-4">
                            @if($car->insurance_included && $car->insurance_cost_per_day)
                                <label class="flex items-start cursor-pointer p-3 sm:p-4 bg-gray-50 rounded-lg hover:bg-gray-100 transition">
                                    <input type="checkbox" name="include_insurance" value="1" 
                                        {{ old('include_insurance') ? 'checked' : '' }}
                                        class="w-4 h-4 sm:w-5 sm:h-5 text-green-600 border-gray-300 rounded focus:ring-green-500 mt-1 flex-shrink-0">
                                    <div class="ml-2 sm:ml-3 flex-1">
                                        <span class="font-semibold text-gray-800 text-sm sm:text-base">Add Insurance Coverage</span>
                                        <p class="text-xs sm:text-sm text-gray-600 mt-1">₦{{ number_format($car->insurance_cost_per_day) }} per day</p>
                                        <p class="text-xs text-gray-500 mt-1">Comprehensive coverage for peace of mind</p>
                                    </div>
                                </label>
                            @endif

                            <label class="flex items-start cursor-pointer p-3 sm:p-4 bg-gray-50 rounded-lg hover:bg-gray-100 transition">
                                <input type="checkbox" name="with_driver" value="1" 
                                    {{ old('with_driver') ? 'checked' : '' }}
                                    class="w-4 h-4 sm:w-5 sm:h-5 text-green-600 border-gray-300 rounded focus:ring-green-500 mt-1 flex-shrink-0">
                                <div class="ml-2 sm:ml-3 flex-1">
                                    <span class="font-semibold text-gray-800 text-sm sm:text-base">Rent with Driver</span>
                                    <p class="text-xs text-gray-500 mt-1">Professional driver included (additional cost may apply)</p>
                                </div>
                            </label>

                            <label class="flex items-start cursor-pointer p-3 sm:p-4 bg-gray-50 rounded-lg hover:bg-gray-100 transition">
                                <input type="checkbox" name="airport_pickup" value="1" 
                                    {{ old('airport_pickup') ? 'checked' : '' }}
                                    class="w-4 h-4 sm:w-5 sm:h-5 text-green-600 border-gray-300 rounded focus:ring-green-500 mt-1 flex-shrink-0">
                                <div class="ml-2 sm:ml-3 flex-1">
                                    <span class="font-semibold text-gray-800 text-sm sm:text-base">Airport Pickup</span>
                                    <p class="text-xs text-gray-500 mt-1">We'll pick you up from the airport (additional cost may apply)</p>
                                </div>
                            </label>
                        </div>
                    </div>

                    <!-- Special Requests -->
                    <div class="bg-white rounded-lg sm:rounded-xl shadow-lg p-4 sm:p-6">
                        <h2 class="text-lg sm:text-xl font-bold text-gray-800 mb-3 sm:mb-4 flex items-center">
                            <i class="fas fa-comment text-green-600 mr-2 sm:mr-3"></i>Special Requests
                        </h2>
                        <textarea name="special_requests" rows="4" 
                            placeholder="Any special requirements or requests?"
                            class="w-full border-2 border-gray-300 rounded-lg px-3 sm:px-4 py-2 sm:py-3 text-sm sm:text-base focus:border-green-600 focus:outline-none">{{ old('special_requests') }}</textarea>
                    </div>
                </div>

                <!-- Booking Summary -->
                <div class="lg:col-span-1">
                    <div class="bg-white rounded-lg sm:rounded-xl shadow-lg p-4 sm:p-6 lg:sticky lg:top-4">
                        <h2 class="text-lg sm:text-xl font-bold text-gray-800 mb-4 sm:mb-6">Booking Summary</h2>

                        <!-- Car Info -->
                        <div class="mb-4 sm:mb-6 pb-4 sm:pb-6 border-b">
                            @if($car->main_photo)
                                <img src="{{ asset('storage/' . $car->main_photo) }}" 
                                    alt="{{ $car->full_name }}"
                                    class="w-full h-28 sm:h-32 object-cover rounded-lg mb-2 sm:mb-3">
                            @endif
                            <h3 class="font-bold text-gray-800 text-sm sm:text-base">{{ $car->full_name }}</h3>
                            <p class="text-xs sm:text-sm text-gray-600">{{ $car->category->name }}</p>
                        </div>

                        <!-- Pricing Breakdown -->
                        <div class="space-y-2 sm:space-y-3 mb-4 sm:mb-6 pb-4 sm:pb-6 border-b">
                            <div class="flex justify-between text-xs sm:text-sm">
                                <span class="text-gray-600">Daily Rate</span>
                                <span class="font-semibold">₦{{ number_format($car->price_per_day) }}</span>
                            </div>
                            @if($car->security_deposit > 0)
                                <div class="flex justify-between text-xs sm:text-sm">
                                    <span class="text-gray-600">Security Deposit</span>
                                    <span class="font-semibold">₦{{ number_format($car->security_deposit) }}</span>
                                </div>
                            @endif
                            @if($car->insurance_cost_per_day)
                                <div class="flex justify-between text-xs sm:text-sm text-gray-500">
                                    <span>Insurance (optional)</span>
                                    <span>₦{{ number_format($car->insurance_cost_per_day) }}/day</span>
                                </div>
                            @endif
                        </div>

                        <!-- Important Info -->
                        <div class="bg-yellow-50 border-l-4 border-yellow-500 p-3 sm:p-4 mb-4 sm:mb-6">
                            <p class="text-xs text-yellow-800">
                                <i class="fas fa-info-circle mr-1"></i>
                                <strong>Note:</strong> Security deposit is refundable after vehicle inspection upon return.
                            </p>
                        </div>

                        <!-- Submit Button -->
                        <button type="submit" 
                            class="w-full bg-gradient-to-r from-green-600 to-teal-600 text-white px-4 sm:px-6 py-3 sm:py-4 rounded-lg hover:shadow-xl transition transform hover:scale-105 font-bold text-sm sm:text-base">
                            <i class="fas fa-check-circle mr-2"></i>Proceed to Payment
                        </button>

                        <p class="text-xs text-gray-500 text-center mt-2 sm:mt-3">
                            <i class="fas fa-lock mr-1"></i>
                            Secure booking process
                        </p>
                    </div>
                </div>
            </div>
        </form>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const minRentalDays = {{ $car->minimum_rental_days }};
    const maxRentalDays = {{ $car->maximum_rental_days ?? 'null' }};
    const pickupDateInput = document.querySelector('input[name="pickup_date"]');
    const returnDateInput = document.querySelector('input[name="return_date"]');
    const warningDiv = document.getElementById('rental-days-warning');
    const warningMessage = document.getElementById('warning-message');
    const submitButton = document.querySelector('button[type="submit"]');

    console.log('Rental validation loaded - Min:', minRentalDays, 'Max:', maxRentalDays, 'Type:', typeof maxRentalDays);

    function validateRentalPeriod() {
        const pickupDate = new Date(pickupDateInput.value);
        const returnDate = new Date(returnDateInput.value);

        if (pickupDateInput.value && returnDateInput.value && returnDate > pickupDate) {
            const diffTime = Math.abs(returnDate - pickupDate);
            const diffDays = Math.ceil(diffTime / (1000 * 60 * 60 * 24));

            console.log('Validating rental period:', diffDays, 'days');

            if (diffDays < minRentalDays) {
                warningMessage.textContent = `Rental period must be at least ${minRentalDays} days. Currently selected: ${diffDays} days.`;
                warningDiv.classList.remove('hidden');
                submitButton.disabled = true;
                submitButton.classList.add('opacity-50', 'cursor-not-allowed');
                return false;
            } else if (maxRentalDays !== null && maxRentalDays > 0 && diffDays > maxRentalDays) {
                warningMessage.textContent = `Rental period cannot exceed ${maxRentalDays} days. Currently selected: ${diffDays} days. Please contact vendor for longer rentals.`;
                warningDiv.classList.remove('hidden');
                submitButton.disabled = true;
                submitButton.classList.add('opacity-50', 'cursor-not-allowed');
                return false;
            } else {
                warningDiv.classList.add('hidden');
                submitButton.disabled = false;
                submitButton.classList.remove('opacity-50', 'cursor-not-allowed');
                return true;
            }
        }
        return true;
    }

    if (pickupDateInput && returnDateInput) {
        pickupDateInput.addEventListener('change', validateRentalPeriod);
        returnDateInput.addEventListener('change', validateRentalPeriod);
    }

    // Validate on form submit
    const form = document.querySelector('form');
    if (form) {
        form.addEventListener('submit', function(e) {
            if (!validateRentalPeriod()) {
                e.preventDefault();
                const maxMsg = maxRentalDays ? ' and ' + maxRentalDays : '';
                alert('Please select a valid rental period (minimum ' + minRentalDays + maxMsg + ' days).');
                warningDiv.scrollIntoView({ behavior: 'smooth', block: 'center' });
            }
        });
    }
});
</script>
@endsection