@extends('layouts.admin')

@section('title', 'Admin Users')

@section('content')
<div>
    <div class="flex justify-between items-center mb-6">
        <h1 class="text-3xl font-bold">Admin Users</h1>
        <button onclick="document.getElementById('create-modal').classList.remove('hidden')"
            class="bg-primary text-white px-6 py-3 rounded hover:bg-primary-dark font-semibold">
            <i class="fas fa-user-plus mr-2"></i> Create Admin User
        </button>
    </div>

    <!-- Users Table -->
    <div class="bg-white rounded-lg shadow-md overflow-hidden">
        <table class="w-full">
            <thead class="bg-light-bg border-b">
                <tr>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">User</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Role</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Created</th>
                    <th class="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">Actions</th>
                </tr>
            </thead>
            <tbody class="divide-y divide-gray-200">
                @forelse($users as $user)
                    <tr class="{{ !$user->email_verified_at ? 'bg-light-bg opacity-60' : '' }}">
                        <td class="px-6 py-4">
                            <div>
                                <div class="font-semibold text-secondary">{{ $user->name }}</div>
                                <div class="text-sm text-gray-600">{{ $user->email }}</div>
                                <div class="text-sm text-gray-500">{{ $user->phone }}</div>
                            </div>
                        </td>
                        <td class="px-6 py-4">
                            @foreach($user->roles as $role)
                                <span class="inline-block px-3 py-1 rounded-full text-xs font-semibold
                                    {{ $role->name === 'owner' ? 'bg-primary/10 text-purple-800' : 
                                       ($role->name === 'manager' ? 'bg-primary/10 text-blue-800' : 'bg-green-100 text-green-800') }}">
                                    {{ ucfirst($role->name) }}
                                </span>
                            @endforeach
                        </td>
                        <td class="px-6 py-4">
                            @if($user->email_verified_at)
                                <span class="inline-block px-3 py-1 bg-green-100 text-green-800 rounded-full text-xs font-semibold">
                                    Active
                                </span>
                            @else
                                <span class="inline-block px-3 py-1 bg-red-100 text-red-800 rounded-full text-xs font-semibold">
                                    Disabled
                                </span>
                            @endif
                        </td>
                        <td class="px-6 py-4 text-sm text-gray-600">
                            {{ $user->created_at->format('M d, Y') }}
                        </td>
                        <td class="px-6 py-4 text-right">
                            <div class="flex justify-end gap-2">
                                @if(!$user->hasRole('owner'))
                                    <button onclick="openEditModal({{ $user->id }}, '{{ $user->name }}', '{{ $user->email }}', '{{ $user->phone }}', '{{ $user->getRoleNames()->first() }}')"
                                        class="text-primary hover:text-blue-800">
                                        <i class="fas fa-edit"></i>
                                    </button>
                                    
                                    @if($user->id !== auth()->id())
                                        <form method="POST" action="{{ route('admin.users.toggle', $user) }}">
                                            @csrf
                                            <button type="submit" 
                                                class="{{ $user->email_verified_at ? 'text-yellow-600 hover:text-yellow-800' : 'text-green-600 hover:text-green-800' }}"
                                                title="{{ $user->email_verified_at ? 'Disable' : 'Enable' }}">
                                                <i class="fas fa-{{ $user->email_verified_at ? 'ban' : 'check-circle' }}"></i>
                                            </button>
                                        </form>
                                        
                                        <form method="POST" action="{{ route('admin.users.destroy', $user) }}"
                                            onsubmit="return confirm('Are you sure you want to delete this user?')">
                                            @csrf
                                            @method('DELETE')
                                            <button type="submit" class="text-red-600 hover:text-red-800">
                                                <i class="fas fa-trash"></i>
                                            </button>
                                        </form>
                                    @endif
                                @else
                                    <span class="text-gray-400 text-sm">Protected</span>
                                @endif
                            </div>
                        </td>
                    </tr>
                @empty
                    <tr>
                        <td colspan="5" class="px-6 py-12 text-center text-gray-500">
                            No admin users found
                        </td>
                    </tr>
                @endforelse
            </tbody>
        </table>
    </div>

    <div class="mt-6">
        {{ $users->links() }}
    </div>
</div>

<!-- Create Modal -->
<div id="create-modal" class="hidden fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
    <div class="bg-white rounded-lg p-8 max-w-md w-full mx-4">
        <div class="flex justify-between items-center mb-6">
            <h2 class="text-2xl font-bold">Create Admin User</h2>
            <button onclick="document.getElementById('create-modal').classList.add('hidden')" 
                class="text-gray-500 hover:text-gray-700">
                <i class="fas fa-times text-2xl"></i>
            </button>
        </div>

        <form method="POST" action="{{ route('admin.users.store') }}">
            @csrf

            <div class="mb-4">
                <label class="block text-gray-700 mb-2">Full Name *</label>
                <input type="text" name="name" value="{{ old('name') }}" required
                    class="w-full border rounded px-4 py-2 @error('name') border-red-500 @enderror">
                @error('name')
                    <p class="text-red-500 text-sm mt-1">{{ $message }}</p>
                @enderror
            </div>

            <div class="mb-4">
                <label class="block text-gray-700 mb-2">Email *</label>
                <input type="email" name="email" value="{{ old('email') }}" required
                    class="w-full border rounded px-4 py-2 @error('email') border-red-500 @enderror">
                @error('email')
                    <p class="text-red-500 text-sm mt-1">{{ $message }}</p>
                @enderror
            </div>

            <div class="mb-4">
                <label class="block text-gray-700 mb-2">Phone *</label>
                <input type="tel" name="phone" value="{{ old('phone') }}" required
                    placeholder="+234..."
                    class="w-full border rounded px-4 py-2 @error('phone') border-red-500 @enderror">
                @error('phone')
                    <p class="text-red-500 text-sm mt-1">{{ $message }}</p>
                @enderror
            </div>

            <div class="mb-4">
                <label class="block text-gray-700 mb-2">Role *</label>
                <select name="role" required
                    class="w-full border rounded px-4 py-2 @error('role') border-red-500 @enderror">
                    <option value="">Select Role</option>
                    <option value="manager">Manager</option>
                    <option value="moderator">Moderator</option>
                </select>
                @error('role')
                    <p class="text-red-500 text-sm mt-1">{{ $message }}</p>
                @enderror
            </div>

            <div class="mb-4">
                <label class="block text-gray-700 mb-2">Password *</label>
                <input type="password" name="password" required
                    class="w-full border rounded px-4 py-2 @error('password') border-red-500 @enderror">
                @error('password')
                    <p class="text-red-500 text-sm mt-1">{{ $message }}</p>
                @enderror
            </div>

            <div class="mb-6">
                <label class="block text-gray-700 mb-2">Confirm Password *</label>
                <input type="password" name="password_confirmation" required
                    class="w-full border rounded px-4 py-2">
            </div>

            <div class="flex gap-3">
                <button type="button" 
                    onclick="document.getElementById('create-modal').classList.add('hidden')"
                    class="flex-1 bg-gray-200 text-gray-700 px-4 py-3 rounded hover:bg-gray-300 font-semibold">
                    Cancel
                </button>
                <button type="submit" 
                    class="flex-1 bg-primary text-white px-4 py-3 rounded hover:bg-primary-dark font-semibold">
                    Create User
                </button>
            </div>
        </form>
    </div>
</div>

<!-- Edit Modal -->
<div id="edit-modal" class="hidden fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
    <div class="bg-white rounded-lg p-8 max-w-md w-full mx-4">
        <div class="flex justify-between items-center mb-6">
            <h2 class="text-2xl font-bold">Edit Admin User</h2>
            <button onclick="document.getElementById('edit-modal').classList.add('hidden')" 
                class="text-gray-500 hover:text-gray-700">
                <i class="fas fa-times text-2xl"></i>
            </button>
        </div>

        <form id="edit-form" method="POST">
            @csrf
            @method('PUT')

            <div class="mb-4">
                <label class="block text-gray-700 mb-2">Full Name *</label>
                <input type="text" id="edit-name" name="name" required
                    class="w-full border rounded px-4 py-2">
            </div>

            <div class="mb-4">
                <label class="block text-gray-700 mb-2">Email *</label>
                <input type="email" id="edit-email" name="email" required
                    class="w-full border rounded px-4 py-2">
            </div>

            <div class="mb-4">
                <label class="block text-gray-700 mb-2">Phone *</label>
                <input type="tel" id="edit-phone" name="phone" required
                    class="w-full border rounded px-4 py-2">
            </div>

            <div class="mb-4">
                <label class="block text-gray-700 mb-2">Role *</label>
                <select id="edit-role" name="role" required
                    class="w-full border rounded px-4 py-2">
                    <option value="manager">Manager</option>
                    <option value="moderator">Moderator</option>
                </select>
            </div>

            <div class="mb-4">
                <label class="block text-gray-700 mb-2">New Password (leave blank to keep current)</label>
                <input type="password" name="password"
                    class="w-full border rounded px-4 py-2">
            </div>

            <div class="mb-6">
                <label class="block text-gray-700 mb-2">Confirm New Password</label>
                <input type="password" name="password_confirmation"
                    class="w-full border rounded px-4 py-2">
            </div>

            <div class="flex gap-3">
                <button type="button" 
                    onclick="document.getElementById('edit-modal').classList.add('hidden')"
                    class="flex-1 bg-gray-200 text-gray-700 px-4 py-3 rounded hover:bg-gray-300 font-semibold">
                    Cancel
                </button>
                <button type="submit" 
                    class="flex-1 bg-primary text-white px-4 py-3 rounded hover:bg-primary-dark font-semibold">
                    Update User
                </button>
            </div>
        </form>
    </div>
</div>

<script>
function openEditModal(id, name, email, phone, role) {
    document.getElementById('edit-form').action = `/admin/users/${id}`;
    document.getElementById('edit-name').value = name;
    document.getElementById('edit-email').value = email;
    document.getElementById('edit-phone').value = phone;
    document.getElementById('edit-role').value = role;
    document.getElementById('edit-modal').classList.remove('hidden');
}
</script>
@endsection
