<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        Schema::create('car_bookings', function (Blueprint $table) {
            $table->id();
            $table->string('booking_reference')->unique();
            $table->foreignId('car_id')->constrained()->onDelete('cascade');
            $table->foreignId('user_id')->constrained()->onDelete('cascade');
            $table->foreignId('vendor_id')->constrained()->onDelete('cascade');
            
            // Booking Details
            $table->date('pickup_date');
            $table->time('pickup_time');
            $table->date('return_date');
            $table->time('return_time');
            $table->integer('total_days');
            
            // Pricing Breakdown
            $table->decimal('rental_cost', 10, 2);
            $table->decimal('insurance_cost', 10, 2)->default(0);
            $table->decimal('security_deposit', 10, 2)->default(0);
            $table->decimal('extra_charges', 10, 2)->default(0); // Late return, extra mileage, etc.
            $table->decimal('discount_amount', 10, 2)->default(0);
            $table->string('promo_code')->nullable();
            $table->decimal('total_amount', 10, 2);
            
            // Driver Information
            $table->string('driver_name');
            $table->string('driver_phone');
            $table->string('driver_email');
            $table->string('driver_license_number');
            $table->date('driver_license_expiry');
            $table->integer('driver_age');
            $table->integer('driver_experience_years');
            $table->string('driver_license_photo')->nullable(); // Uploaded license
            
            // Additional Options
            $table->boolean('with_driver')->default(false);
            $table->decimal('driver_cost', 10, 2)->nullable();
            $table->boolean('airport_pickup')->default(false);
            $table->decimal('airport_pickup_cost', 10, 2)->nullable();
            $table->text('special_requests')->nullable();
            
            // Mileage Tracking
            $table->integer('starting_mileage')->nullable();
            $table->integer('ending_mileage')->nullable();
            $table->integer('total_mileage')->nullable();
            $table->integer('allowed_mileage')->nullable();
            $table->integer('extra_mileage')->nullable();
            
            // Payment
            $table->enum('payment_status', ['pending', 'pending_verification', 'paid', 'partially_refunded', 'refunded', 'failed'])->default('pending');
            $table->enum('payment_method', ['bank_transfer', 'paystack', 'flutterwave', 'cash'])->nullable();
            $table->string('payment_reference')->nullable();
            $table->string('payment_proof')->nullable();
            $table->timestamp('paid_at')->nullable();
            $table->timestamp('verified_at')->nullable();
            
            // Deposit Refund
            $table->enum('deposit_status', ['held', 'refunded', 'deducted'])->default('held');
            $table->decimal('deposit_refund_amount', 10, 2)->nullable();
            $table->text('deposit_deduction_reason')->nullable();
            $table->timestamp('deposit_refunded_at')->nullable();
            
            // Status
            $table->enum('status', ['pending', 'confirmed', 'active', 'completed', 'cancelled'])->default('pending');
            $table->text('cancellation_reason')->nullable();
            $table->timestamp('cancelled_at')->nullable();
            $table->timestamp('confirmed_at')->nullable();
            $table->timestamp('pickup_at')->nullable();
            $table->timestamp('returned_at')->nullable();
            
            // Vehicle Condition
            $table->text('pickup_condition_notes')->nullable();
            $table->json('pickup_photos')->nullable();
            $table->text('return_condition_notes')->nullable();
            $table->json('return_photos')->nullable();
            $table->text('damage_report')->nullable();
            
            $table->timestamps();
            
            // Indexes
            $table->index(['user_id', 'status']);
            $table->index(['vendor_id', 'status']);
            $table->index(['car_id', 'pickup_date', 'return_date']);
            $table->index('payment_status');
        });
    }

    public function down(): void
    {
        Schema::dropIfExists('car_bookings');
    }
};
