<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        Schema::create('cars', function (Blueprint $table) {
            $table->id();
            $table->foreignId('vendor_id')->constrained()->onDelete('cascade');
            $table->foreignId('car_category_id')->constrained()->onDelete('cascade');
            
            // Basic Information
            $table->string('brand'); // Toyota, Honda, Mercedes, etc.
            $table->string('model'); // Camry, Accord, E-Class, etc.
            $table->integer('year');
            $table->string('plate_number')->unique();
            $table->string('color');
            $table->enum('transmission', ['automatic', 'manual']);
            $table->enum('fuel_type', ['petrol', 'diesel', 'electric', 'hybrid']);
            
            // Capacity & Features
            $table->integer('seats');
            $table->integer('doors');
            $table->integer('luggage_capacity'); // Number of large bags
            $table->boolean('air_conditioning')->default(true);
            $table->json('features')->nullable(); // GPS, Bluetooth, Child seat, etc.
            
            // Location
            $table->string('state');
            $table->string('city');
            $table->text('pickup_location'); // Detailed address
            $table->decimal('latitude', 10, 8)->nullable();
            $table->decimal('longitude', 11, 8)->nullable();
            
            // Pricing
            $table->decimal('price_per_day', 10, 2);
            $table->decimal('price_per_week', 10, 2)->nullable();
            $table->decimal('price_per_month', 10, 2)->nullable();
            $table->decimal('security_deposit', 10, 2)->default(0);
            $table->integer('minimum_rental_days')->default(1);
            $table->integer('maximum_rental_days')->nullable();
            
            // Insurance & Policies
            $table->boolean('insurance_included')->default(false);
            $table->decimal('insurance_cost_per_day', 10, 2)->nullable();
            $table->integer('mileage_limit_per_day')->nullable(); // km per day
            $table->decimal('extra_mileage_cost', 10, 2)->nullable(); // cost per extra km
            $table->text('cancellation_policy')->nullable();
            $table->text('rental_terms')->nullable();
            
            // Requirements
            $table->integer('minimum_driver_age')->default(21);
            $table->integer('minimum_license_years')->default(1);
            $table->boolean('international_license_accepted')->default(false);
            
            // Description & Images
            $table->text('description')->nullable();
            $table->json('photos')->nullable();
            
            // Availability
            $table->boolean('is_available')->default(true);
            $table->enum('status', ['pending', 'approved', 'rejected'])->default('pending');
            $table->text('rejection_reason')->nullable();
            
            // Stats
            $table->integer('total_bookings')->default(0);
            $table->decimal('average_rating', 3, 2)->default(0);
            $table->integer('views')->default(0);
            
            $table->timestamps();
            $table->softDeletes();
            
            // Indexes
            $table->index(['vendor_id', 'status']);
            $table->index(['state', 'city']);
            $table->index(['car_category_id', 'is_available']);
        });
    }

    public function down(): void
    {
        Schema::dropIfExists('cars');
    }
};
