<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        Schema::create('bookings', function (Blueprint $table) {
            $table->id();
            $table->foreignId('user_id')->constrained()->onDelete('cascade');
            $table->foreignId('property_id')->constrained()->onDelete('cascade');
            $table->foreignId('vendor_id')->constrained()->onDelete('cascade');
            
            // Dates
            $table->date('start_date');
            $table->date('end_date');
            $table->integer('nights_count');
            
            // Guest info
            $table->integer('adults_count')->default(1);
            $table->integer('children_count')->default(0);
            $table->integer('guest_count'); // Total
            $table->enum('trip_type', ['normal', 'party'])->default('normal');
            
            // Approval flow
            $table->boolean('requires_vendor_approval')->default(false);
            $table->enum('vendor_approval_status', ['pending', 'approved', 'rejected'])->default('pending');
            $table->timestamp('vendor_approved_at')->nullable();
            $table->text('vendor_rejection_reason')->nullable();
            
            // Payment
            $table->decimal('property_price_per_night', 10, 2);
            $table->decimal('subtotal', 10, 2); // nights * price
            $table->decimal('caution_fee_amount', 10, 2)->default(0);
            $table->decimal('total_amount', 10, 2);
            $table->enum('payment_status', ['pending', 'paid', 'refunded', 'partially_refunded'])->default('pending');
            $table->string('payment_reference')->nullable()->unique();
            $table->string('payment_method')->nullable(); // paystack, flutterwave
            $table->timestamp('paid_at')->nullable();
            
            // Admin verification
            $table->boolean('admin_verified')->default(false);
            $table->foreignId('verified_by')->nullable()->constrained('users')->nullOnDelete();
            $table->timestamp('admin_verified_at')->nullable();
            
            // Caution refund
            $table->enum('caution_refund_status', ['pending', 'approved', 'rejected', 'refunded'])->default('pending');
            $table->timestamp('caution_refund_decided_at')->nullable();
            $table->foreignId('caution_decided_by')->nullable()->constrained('users')->nullOnDelete();
            $table->text('caution_refund_reason')->nullable();
            
            // Booking status
            $table->enum('booking_status', ['pending', 'confirmed', 'cancelled', 'completed'])->default('pending');
            $table->timestamp('cancelled_at')->nullable();
            $table->text('cancellation_reason')->nullable();
            
            // Visibility flags
            $table->boolean('user_visible_vendor_info')->default(false);
            $table->boolean('vendor_visible_user_info')->default(false);
            $table->boolean('map_visible')->default(false);
            
            // Commission tracking
            $table->decimal('commission_rate', 5, 2);
            $table->decimal('commission_amount', 10, 2)->default(0);
            $table->decimal('vendor_payout_amount', 10, 2)->default(0);
            
            $table->timestamps();
            $table->softDeletes();
            
            $table->index(['user_id', 'booking_status']);
            $table->index(['vendor_id', 'vendor_approval_status']);
            $table->index(['payment_status', 'admin_verified']);
            $table->index(['start_date', 'end_date']);
        });
    }

    public function down(): void
    {
        Schema::dropIfExists('bookings');
    }
};