<?php

namespace App\Services\Payment;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class FlutterwaveService implements PaymentServiceInterface
{
    protected ?string $secretKey;
    protected ?string $publicKey;
    protected string $baseUrl = 'https://api.flutterwave.com/v3';
    
    public function __construct()
    {
        $this->secretKey = config('services.flutterwave.secret_key');
        $this->publicKey = config('services.flutterwave.public_key');
    }
    
    protected function isConfigured(): bool
    {
        return !empty($this->secretKey) && !empty($this->publicKey);
    }
    
    public function initializePayment(array $data): array
    {
        if (!$this->isConfigured()) {
            Log::warning('Flutterwave is not configured');
            throw new \Exception('Payment service is not configured. Please contact support.');
        }
        
        $response = Http::withToken($this->secretKey)
            ->post("{$this->baseUrl}/payments", [
                'tx_ref' => $data['reference'],
                'amount' => $data['amount'],
                'currency' => 'NGN',
                'redirect_url' => $data['callback_url'] ?? config('app.url') . '/payment/callback',
                'customer' => [
                    'email' => $data['email'],
                    'phone_number' => $data['phone'] ?? null,
                    'name' => $data['name'] ?? null,
                ],
                'customizations' => [
                    'title' => 'ShortStayNG Payment',
                    'description' => $data['description'] ?? 'Property Booking Payment',
                    'logo' => config('app.url') . '/images/logo.png',
                ],
                'meta' => $data['metadata'] ?? [],
            ]);
            
        if (!$response->successful()) {
            Log::error('Flutterwave initialization failed', [
                'response' => $response->json(),
                'data' => $data,
            ]);
            
            throw new \Exception('Payment initialization failed');
        }
        
        $result = $response->json();
        
        return [
            'status' => 'success',
            'payment_url' => $result['data']['link'],
            'reference' => $data['reference'],
        ];
    }
    
    public function verifyPayment(string $reference): array
    {
        if (!$this->isConfigured()) {
            Log::warning('Flutterwave is not configured');
            throw new \Exception('Payment service is not configured. Please contact support.');
        }
        
        $response = Http::withToken($this->secretKey)
            ->get("{$this->baseUrl}/transactions", [
                'tx_ref' => $reference,
            ]);
            
        if (!$response->successful()) {
            Log::error('Flutterwave verification failed', [
                'reference' => $reference,
                'response' => $response->json(),
            ]);
            
            throw new \Exception('Payment verification failed');
        }
        
        $result = $response->json();
        $data = $result['data'][0] ?? null;
        
        if (!$data) {
            throw new \Exception('Transaction not found');
        }
        
        return [
            'status' => $data['status'],
            'reference' => $data['tx_ref'],
            'amount' => $data['amount'],
            'currency' => $data['currency'],
            'paid_at' => $data['created_at'],
            'customer' => [
                'email' => $data['customer']['email'],
                'phone' => $data['customer']['phone_number'] ?? null,
            ],
            'metadata' => $data['meta'] ?? [],
            'gateway_response' => $result,
        ];
    }
    
    public function verifyWebhook(array $payload, string $signature): bool
    {
        $secret = config('services.flutterwave.webhook_secret');
        
        return hash_equals($secret, $signature);
    }
}
