<?php

namespace App\Services\Notification;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class WhatsAppBusinessService implements WhatsAppServiceInterface
{
    protected string $apiUrl;
    protected string $token;
    protected string $phoneNumberId;
    
    public function __construct()
    {
        $this->apiUrl = config('services.whatsapp_business.api_url');
        $this->token = config('services.whatsapp_business.token');
        $this->phoneNumberId = config('services.whatsapp_business.phone_number_id');
    }
    
    public function sendMessage(string $to, string $message): bool
    {
        try {
            $to = $this->formatPhoneNumber($to);
            
            $response = Http::withToken($this->token)
                ->post("{$this->apiUrl}/{$this->phoneNumberId}/messages", [
                    'messaging_product' => 'whatsapp',
                    'to' => $to,
                    'type' => 'text',
                    'text' => [
                        'body' => $message,
                    ],
                ]);
                
            if (!$response->successful()) {
                Log::error('WhatsApp Business API failed', [
                    'to' => $to,
                    'response' => $response->json(),
                ]);
                return false;
            }
            
            return true;
        } catch (\Exception $e) {
            Log::error('WhatsApp Business API exception', [
                'to' => $to,
                'error' => $e->getMessage(),
            ]);
            return false;
        }
    }
    
    public function sendTemplate(string $to, string $templateName, array $params): bool
    {
        try {
            $to = $this->formatPhoneNumber($to);
            
            $response = Http::withToken($this->token)
                ->post("{$this->apiUrl}/{$this->phoneNumberId}/messages", [
                    'messaging_product' => 'whatsapp',
                    'to' => $to,
                    'type' => 'template',
                    'template' => [
                        'name' => $templateName,
                        'language' => [
                            'code' => 'en',
                        ],
                        'components' => $this->buildTemplateComponents($params),
                    ],
                ]);
                
            if (!$response->successful()) {
                Log::error('WhatsApp Business template failed', [
                    'to' => $to,
                    'template' => $templateName,
                    'response' => $response->json(),
                ]);
                
                // Fallback to text message
                return $this->sendMessage($to, $this->buildFallbackMessage($templateName, $params));
            }
            
            return true;
        } catch (\Exception $e) {
            Log::error('WhatsApp Business template exception', [
                'to' => $to,
                'error' => $e->getMessage(),
            ]);
            
            // Fallback to text message
            return $this->sendMessage($to, $this->buildFallbackMessage($templateName, $params));
        }
    }
    
    protected function formatPhoneNumber(string $phone): string
    {
        $phone = preg_replace('/\D/', '', $phone);
        
        if (!str_starts_with($phone, '234')) {
            if (str_starts_with($phone, '0')) {
                $phone = '234' . substr($phone, 1);
            } else {
                $phone = '234' . $phone;
            }
        }
        
        return $phone;
    }
    
    protected function buildTemplateComponents(array $params): array
    {
        $components = [];
        
        if (!empty($params['header'])) {
            $components[] = [
                'type' => 'header',
                'parameters' => $params['header'],
            ];
        }
        
        if (!empty($params['body'])) {
            $components[] = [
                'type' => 'body',
                'parameters' => $params['body'],
            ];
        }
        
        return $components;
    }
    
    protected function buildFallbackMessage(string $templateName, array $params): string
    {
        return match($templateName) {
            'verification' => "🏨 ShortStayNG\n\nYour verification code is: {$params['code']}\n\nThis code expires in 10 minutes.",
            'booking_request' => "🏨 ShortStayNG\n\nNew booking request! Please check your dashboard.",
            default => "ShortStayNG: " . ($params['message'] ?? 'Notification'),
        };
    }
}