<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class BusinessVerificationService
{
    private $baseUrl;
    private $token;

    public function __construct()
    {
        $this->baseUrl = config('services.business_verification.base_url');
        $this->token = config('services.business_verification.token');
    }

    /**
     * Verify a business registration number
     *
     * @param string $registrationNumber
     * @return array
     */
    public function verifyBusiness(string $registrationNumber): array
    {
        try {
            // Format registration number (remove spaces and ensure prefix)
            $formattedNumber = $this->formatRegistrationNumber($registrationNumber);

            Log::info('Calling business verification API', [
                'original_number' => $registrationNumber,
                'formatted_number' => $formattedNumber,
            ]);

            $response = Http::timeout(30)
                ->withHeaders([
                    'Authorization' => 'Bearer ' . $this->token,
                    'Content-Type' => 'application/json',
                    'Accept' => 'application/json',
                ])
                ->post($this->baseUrl . '/v2/api/verifications/ng/company/basic', [
                    'registrationNumber' => $formattedNumber,
                    'isConsent' => true,
                ]);

            // Log raw response for debugging
            Log::info('Business verification API raw response', [
                'status' => $response->status(),
                'body' => $response->body(),
            ]);

            if ($response->successful()) {
                $data = $response->json();
                
                // Check if response has expected structure
                if (!isset($data['success']) || !isset($data['data'])) {
                    Log::error('Unexpected API response structure', ['data' => $data]);
                    return [
                        'success' => false,
                        'status' => 'error',
                        'message' => 'Unexpected response format from verification API',
                        'data' => null,
                    ];
                }

                // API returned success
                if ($data['success']) {
                    return [
                        'success' => true,
                        'status' => $data['data']['status'] ?? 'unknown',
                        'data' => $data['data'] ?? null,
                        'message' => $data['message'] ?? 'Verification completed',
                    ];
                }

                // API returned success=false
                return [
                    'success' => false,
                    'status' => 'error',
                    'message' => $data['message'] ?? 'Verification failed',
                    'data' => $data['data'] ?? null,
                ];
            }

            // HTTP error (4xx, 5xx)
            $errorBody = $response->json();
            Log::error('Business verification API HTTP error', [
                'status' => $response->status(),
                'body' => $errorBody,
            ]);

            return [
                'success' => false,
                'status' => 'error',
                'message' => $errorBody['message'] ?? 'API request failed with status ' . $response->status(),
                'data' => null,
            ];

        } catch (\Illuminate\Http\Client\ConnectionException $e) {
            Log::error('Business verification connection error', [
                'message' => $e->getMessage(),
            ]);

            return [
                'success' => false,
                'status' => 'error',
                'message' => 'Unable to connect to verification service. Please check your internet connection or try again later.',
                'data' => null,
            ];

        } catch (\Exception $e) {
            Log::error('Business verification exception', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);

            return [
                'success' => false,
                'status' => 'error',
                'message' => 'An unexpected error occurred: ' . $e->getMessage(),
                'data' => null,
            ];
        }
    }

    /**
     * Format registration number with proper prefix
     *
     * @param string $registrationNumber
     * @return string
     */
    private function formatRegistrationNumber(string $registrationNumber): string
    {
        // Remove spaces and special characters
        $cleaned = preg_replace('/[^A-Za-z0-9]/', '', $registrationNumber);
        
        // Ensure it has a valid prefix
        $validPrefixes = ['RC', 'BN', 'IT', 'LP', 'LLP'];
        $hasValidPrefix = false;
        
        foreach ($validPrefixes as $prefix) {
            if (stripos($cleaned, $prefix) === 0) {
                $hasValidPrefix = true;
                break;
            }
        }
        
        // If no prefix, assume RC (Private Company)
        if (!$hasValidPrefix) {
            $cleaned = 'RC' . $cleaned;
        }
        
        return strtoupper($cleaned);
    }

    /**
     * Validate if business details match submitted information
     *
     * @param array $verificationData
     * @param array $submittedData
     * @return array
     */
    public function validateBusinessMatch(array $verificationData, array $submittedData): array
    {
        $matches = [];
        $mismatches = [];
        $similarity = 0;

        // Check business name similarity
        if (isset($verificationData['name']) && isset($submittedData['business_name'])) {
            similar_text(
                strtolower(trim($verificationData['name'])),
                strtolower(trim($submittedData['business_name'])),
                $similarity
            );

            if ($similarity >= 80) {
                $matches[] = "Business name matches ({$similarity}% similar)";
            } else if ($similarity >= 60) {
                $mismatches[] = "Business name partially matches ({$similarity}% similar): CAC='{$verificationData['name']}', Submitted='{$submittedData['business_name']}'";
            } else {
                $mismatches[] = "Business name mismatch ({$similarity}% similar): CAC='{$verificationData['name']}', Submitted='{$submittedData['business_name']}'";
            }
        }

        // Check company status
        if (isset($verificationData['companyStatus'])) {
            if ($verificationData['companyStatus'] === 'ACTIVE') {
                $matches[] = 'Company is ACTIVE';
            } else {
                $mismatches[] = "Company status is {$verificationData['companyStatus']} (not ACTIVE)";
            }
        }

        // Check registration date exists
        if (isset($verificationData['registrationDate'])) {
            $matches[] = "Registered on {$verificationData['registrationDate']}";
        }

        return [
            'is_valid' => empty($mismatches),
            'matches' => $matches,
            'mismatches' => $mismatches,
            'similarity_score' => round($similarity, 2),
        ];
    }
}