<?php

namespace App\Notifications;

use Illuminate\Bus\Queueable;
use Illuminate\Notifications\Notification;
use Illuminate\Notifications\Messages\MailMessage;
use App\Models\Booking;
use App\Services\Notification\WhatsAppFactory;

class BookingNotification extends Notification
{
    use Queueable;

    public function __construct(private Booking $booking, private string $type)
    {
    }

    public function via($notifiable): array
    {
        $channels = ['mail', 'database'];
        
        // Only add WhatsApp if configured and user has phone number
        if ($this->isWhatsAppAvailable($notifiable)) {
            $channels[] = 'whatsapp';
        }
        
        return $channels;
    }
    
    protected function isWhatsAppAvailable($notifiable): bool
    {
        // Check if WhatsApp is configured
        $driver = config('services.whatsapp.driver');
        if (!$driver) {
            return false;
        }
        
        // Check if user has phone number
        if (empty($notifiable->phone)) {
            return false;
        }
        
        // Check if required credentials are set based on driver
        if ($driver === 'twilio') {
            return !empty(config('services.twilio.sid')) && 
                   !empty(config('services.twilio.token'));
        }
        
        if ($driver === 'business') {
            return !empty(config('services.whatsapp_business.token'));
        }
        
        return false;
    }

    public function toMail($notifiable): MailMessage
    {
        return match($this->type) {
            'created' => $this->bookingCreatedMail($notifiable),
            'approved' => $this->bookingApprovedMail($notifiable),
            'rejected' => $this->bookingRejectedMail($notifiable),
            'payment_confirmed' => $this->paymentConfirmedMail($notifiable),
            default => (new MailMessage)->line('Booking update'),
        };
    }

    protected function bookingCreatedMail($notifiable): MailMessage
    {
        return (new MailMessage)
            ->subject('ShortStayNG - New Booking Request')
            ->greeting('Hello ' . $notifiable->name . '!')
            ->line('You have received a new booking request for: ' . $this->booking->property->title)
            ->line('Check-in: ' . $this->booking->start_date->format('M d, Y'))
            ->line('Check-out: ' . $this->booking->end_date->format('M d, Y'))
            ->line('Guests: ' . $this->booking->guest_count)
            ->action('View Booking', route('vendor.bookings.show', $this->booking))
            ->line('Please review and respond to this request promptly.');
    }

    protected function bookingApprovedMail($notifiable): MailMessage
    {
        return (new MailMessage)
            ->subject('ShortStayNG - Booking Approved!')
            ->greeting('Great news, ' . $notifiable->name . '!')
            ->line('Your booking request has been approved by the vendor.')
            ->line('Property: ' . $this->booking->property->title)
            ->line('Total Amount: ₦' . number_format($this->booking->total_amount))
            ->action('Complete Payment', route('bookings.payment', $this->booking))
            ->line('Please complete your payment to confirm the booking.');
    }

    protected function bookingRejectedMail($notifiable): MailMessage
    {
        return (new MailMessage)
            ->subject('ShortStayNG - Booking Request Update')
            ->greeting('Hello ' . $notifiable->name)
            ->line('Unfortunately, your booking request has been declined by the vendor.')
            ->line('Reason: ' . ($this->booking->vendor_rejection_reason ?? 'Not specified'))
            ->action('Browse Other Properties', route('properties.index'))
            ->line('We apologize for any inconvenience.');
    }

    protected function paymentConfirmedMail($notifiable): MailMessage
    {
        return (new MailMessage)
            ->subject('ShortStayNG - Payment Confirmed!')
            ->greeting('Congratulations, ' . $notifiable->name . '!')
            ->line('Your payment has been confirmed.')
            ->line('Property: ' . $this->booking->property->title)
            ->line('Booking Reference: ' . $this->booking->payment_reference)
            ->line('Your booking details, including contact information and location, are now available.')
            ->action('View Booking Details', route('bookings.show', $this->booking))
            ->line('Have a wonderful stay!');
    }

    public function toWhatsApp($notifiable): bool
    {
        try {
            $service = WhatsAppFactory::create();
            
            $params = match($this->type) {
                'created' => [
                    'property_name' => $this->booking->property->title,
                    'guest_count' => $this->booking->guest_count,
                    'dates' => $this->booking->start_date->format('M d') . ' - ' . $this->booking->end_date->format('M d'),
                ],
                'approved' => [
                    'property_name' => $this->booking->property->title,
                    'payment_link' => route('bookings.payment', $this->booking),
                ],
                'payment_confirmed' => [
                    'reference' => $this->booking->payment_reference,
                    'property_name' => $this->booking->property->title,
                ],
                default => ['message' => 'Booking update'],
            };

            return $service->sendTemplate($notifiable->phone, $this->type, $params);
        } catch (\Exception $e) {
            // Log error but don't fail the notification
            \Log::error('WhatsApp notification failed: ' . $e->getMessage(), [
                'booking_id' => $this->booking->id,
                'user_id' => $notifiable->id,
            ]);
            
            return false;
        }
    }

    public function toArray($notifiable): array
    {
        return [
            'booking_id' => $this->booking->id,
            'type' => $this->type,
            'message' => $this->getNotificationMessage(),
        ];
    }

    protected function getNotificationMessage(): string
    {
        return match($this->type) {
            'created' => 'New booking request received',
            'approved' => 'Your booking has been approved',
            'rejected' => 'Your booking request was declined',
            'payment_confirmed' => 'Payment confirmed successfully',
            default => 'Booking update',
        };
    }
}