<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class PromoCode extends Model
{
    protected $fillable = [
        'code',
        'name',
        'description',
        'discount_type',
        'discount_value',
        'max_discount_amount',
        'min_ad_amount',
        'usage_limit',
        'usage_count',
        'usage_per_vendor',
        'valid_from',
        'valid_until',
        'is_active',
        'created_by',
    ];

    protected $casts = [
        'discount_value' => 'decimal:2',
        'max_discount_amount' => 'decimal:2',
        'min_ad_amount' => 'decimal:2',
        'valid_from' => 'datetime',
        'valid_until' => 'datetime',
        'is_active' => 'boolean',
    ];

    public function creator(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function usages(): HasMany
    {
        return $this->hasMany(PromoCodeUsage::class);
    }

    public function isValid(): bool
    {
        if (!$this->is_active) {
            return false;
        }

        $now = now();

        // Check if valid_from date has passed (if set)
        if ($this->valid_from) {
            $validFrom = \Carbon\Carbon::parse($this->valid_from);
            if ($now->lt($validFrom)) {
                return false;
            }
        }

        // Check if valid_until date has not passed (if set)
        if ($this->valid_until) {
            $validUntil = \Carbon\Carbon::parse($this->valid_until);
            if ($now->gt($validUntil)) {
                return false;
            }
        }

        // Check usage limit
        if ($this->usage_limit && $this->usage_count >= $this->usage_limit) {
            return false;
        }

        return true;
    }

    public function canBeUsedBy(Vendor $vendor): bool
    {
        if (!$this->isValid()) {
            return false;
        }

        $vendorUsageCount = $this->usages()
            ->where('vendor_id', $vendor->id)
            ->count();

        return $vendorUsageCount < $this->usage_per_vendor;
    }

    public function calculateDiscount(float $amount): float
    {
        if ($amount < $this->min_ad_amount) {
            return 0;
        }

        if ($this->discount_type === 'percentage') {
            $discount = ($amount * $this->discount_value) / 100;
            
            if ($this->max_discount_amount) {
                $discount = min($discount, $this->max_discount_amount);
            }
            
            return round($discount, 2);
        }

        // Fixed amount
        return min($this->discount_value, $amount);
    }

    public function scopeActive($query)
    {
        return $query->where('is_active', true)
            ->where(function ($q) {
                $q->whereNull('valid_from')
                    ->orWhere('valid_from', '<=', now());
            })
            ->where(function ($q) {
                $q->whereNull('valid_until')
                    ->orWhere('valid_until', '>=', now());
            });
    }
}
