<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class DiscountDeal extends Model
{
    protected $fillable = [
        'code',
        'name',
        'description',
        'discount_type',
        'discount_value',
        'max_discount_amount',
        'min_booking_amount',
        'usage_limit',
        'usage_count',
        'usage_per_user',
        'valid_from',
        'valid_until',
        'is_active',
        'is_featured',
        'created_by',
        'show_on_homepage',
        'banner_title',
        'banner_description',
        'banner_button_text',
        'banner_button_link',
        'banner_bg_from',
        'banner_bg_to',
        'banner_icon',
    ];

    protected $casts = [
        'discount_value' => 'decimal:2',
        'max_discount_amount' => 'decimal:2',
        'min_booking_amount' => 'decimal:2',
        'valid_from' => 'datetime',
        'valid_until' => 'datetime',
        'is_active' => 'boolean',
        'is_featured' => 'boolean',
        'show_on_homepage' => 'boolean',
    ];

    public function creator(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function usages(): HasMany
    {
        return $this->hasMany(DiscountDealUsage::class);
    }

    public function bookings(): HasMany
    {
        return $this->hasMany(Booking::class);
    }

    public function isValid(): bool
    {
        if (!$this->is_active) {
            return false;
        }

        $now = now();

        // Check if valid_from date has passed (if set)
        if ($this->valid_from) {
            $validFrom = \Carbon\Carbon::parse($this->valid_from);
            if ($now->lt($validFrom)) {
                return false;
            }
        }

        // Check if valid_until date has not passed (if set)
        if ($this->valid_until) {
            $validUntil = \Carbon\Carbon::parse($this->valid_until);
            if ($now->gt($validUntil)) {
                return false;
            }
        }

        // Check usage limit
        if ($this->usage_limit && $this->usage_count >= $this->usage_limit) {
            return false;
        }

        return true;
    }

    public function canBeUsedBy(User $user): bool
    {
        if (!$this->isValid()) {
            return false;
        }

        $userUsageCount = $this->usages()
            ->where('user_id', $user->id)
            ->count();

        return $userUsageCount < $this->usage_per_user;
    }

    public function calculateDiscount(float $amount): float
    {
        if ($amount < $this->min_booking_amount) {
            return 0;
        }

        if ($this->discount_type === 'percentage') {
            $discount = ($amount * $this->discount_value) / 100;
            
            if ($this->max_discount_amount) {
                $discount = min($discount, $this->max_discount_amount);
            }
            
            return round($discount, 2);
        }

        // Fixed amount
        return min($this->discount_value, $amount);
    }

    public function scopeActive($query)
    {
        return $query->where('is_active', true)
            ->where(function ($q) {
                $q->whereNull('valid_from')
                    ->orWhere('valid_from', '<=', now());
            })
            ->where(function ($q) {
                $q->whereNull('valid_until')
                    ->orWhere('valid_until', '>=', now());
            });
    }

    public function scopeFeatured($query)
    {
        return $query->where('is_featured', true)->active();
    }
}
