<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Car extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'vendor_id',
        'car_category_id',
        'brand',
        'model',
        'year',
        'plate_number',
        'color',
        'transmission',
        'fuel_type',
        'seats',
        'doors',
        'luggage_capacity',
        'air_conditioning',
        'features',
        'state',
        'city',
        'pickup_location',
        'latitude',
        'longitude',
        'price_per_day',
        'price_per_week',
        'price_per_month',
        'security_deposit',
        'minimum_rental_days',
        'maximum_rental_days',
        'insurance_included',
        'insurance_cost_per_day',
        'mileage_limit_per_day',
        'extra_mileage_cost',
        'cancellation_policy',
        'rental_terms',
        'minimum_driver_age',
        'minimum_license_years',
        'international_license_accepted',
        'description',
        'photos',
        'is_available',
        'status',
        'rejection_reason',
        'total_bookings',
        'average_rating',
        'views',
    ];

    protected $casts = [
        'features' => 'array',
        'photos' => 'array',
        'air_conditioning' => 'boolean',
        'insurance_included' => 'boolean',
        'international_license_accepted' => 'boolean',
        'is_available' => 'boolean',
        'price_per_day' => 'decimal:2',
        'price_per_week' => 'decimal:2',
        'price_per_month' => 'decimal:2',
        'security_deposit' => 'decimal:2',
        'insurance_cost_per_day' => 'decimal:2',
        'extra_mileage_cost' => 'decimal:2',
        'average_rating' => 'decimal:2',
        'latitude' => 'decimal:8',
        'longitude' => 'decimal:8',
    ];

    // Relationships
    public function vendor()
    {
        return $this->belongsTo(Vendor::class);
    }

    public function category()
    {
        return $this->belongsTo(CarCategory::class, 'car_category_id');
    }

    public function bookings()
    {
        return $this->hasMany(CarBooking::class);
    }

    public function reviews()
    {
        return $this->hasMany(CarReview::class);
    }

    public function approvedReviews()
    {
        return $this->reviews()->where('is_approved', true);
    }

    // Scopes
    public function scopeApproved($query)
    {
        return $query->where('status', 'approved');
    }

    public function scopeAvailable($query)
    {
        return $query->where('is_available', true);
    }

    public function scopeInLocation($query, $state = null, $city = null)
    {
        if ($state) {
            $query->where('state', $state);
        }
        if ($city) {
            $query->where('city', $city);
        }
        return $query;
    }

    public function scopeInCategory($query, $categoryId)
    {
        return $query->where('car_category_id', $categoryId);
    }

    public function scopePriceRange($query, $min = null, $max = null)
    {
        if ($min) {
            $query->where('price_per_day', '>=', $min);
        }
        if ($max) {
            $query->where('price_per_day', '<=', $max);
        }
        return $query;
    }

    // Methods
    public function isAvailableForDates($pickupDate, $returnDate)
    {
        // Check for any bookings that are paid but not yet returned
        // Statuses: confirmed, active (in use), or paid pending pickup
        return !$this->bookings()
            ->where('payment_status', 'paid')
            ->whereIn('status', ['pending', 'confirmed', 'active'])
            ->whereNull('returned_at') // Not yet returned
            ->where(function ($query) use ($pickupDate, $returnDate) {
                $query->whereBetween('pickup_date', [$pickupDate, $returnDate])
                    ->orWhereBetween('return_date', [$pickupDate, $returnDate])
                    ->orWhere(function ($q) use ($pickupDate, $returnDate) {
                        $q->where('pickup_date', '<=', $pickupDate)
                          ->where('return_date', '>=', $returnDate);
                    });
            })
            ->exists();
    }
    
    public function hasActiveBooking()
    {
        return $this->bookings()
            ->where('payment_status', 'paid')
            ->whereIn('status', ['confirmed', 'active'])
            ->whereNull('returned_at')
            ->exists();
    }

    public function calculateRentalCost($days)
    {
        // Calculate base rental cost
        $rentalCost = 0;
        
        // Check for weekly/monthly rates
        if ($days >= 30 && $this->price_per_month) {
            $months = floor($days / 30);
            $remainingDays = $days % 30;
            $rentalCost = ($months * $this->price_per_month) + ($remainingDays * $this->price_per_day);
        } elseif ($days >= 7 && $this->price_per_week) {
            $weeks = floor($days / 7);
            $remainingDays = $days % 7;
            $rentalCost = ($weeks * $this->price_per_week) + ($remainingDays * $this->price_per_day);
        } else {
            $rentalCost = $days * $this->price_per_day;
        }
        
        // Calculate insurance cost (10% of rental cost per day, or custom rate)
        $insuranceCost = ($rentalCost * 0.10); // 10% of rental cost
        
        return [
            'rental_cost' => $rentalCost,
            'insurance_cost' => $insuranceCost,
            'total_days' => $days,
        ];
    }

    public function getFullNameAttribute()
    {
        return "{$this->year} {$this->brand} {$this->model}";
    }

    public function getMainPhotoAttribute()
    {
        return $this->photos[0] ?? null;
    }

    public function incrementViews()
    {
        $this->increment('views');
    }

    public function updateRating()
    {
        $avgRating = $this->approvedReviews()->avg('overall_rating');
        $this->update(['average_rating' => $avgRating ?? 0]);
    }
}
