<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class AdPlan extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'description',
        'placement_type',
        'duration_days',
        'price',
        'active',
    ];

    protected $casts = [
        'active' => 'boolean',
        'price' => 'decimal:2',
    ];

    // Relationships
    public function vendorAds()
    {
        return $this->hasMany(VendorAd::class);
    }

    // Scopes
    public function scopeActive($query)
    {
        return $query->where('active', true);
    }

    public function scopeFeatured($query)
    {
        return $query->where('placement_type', 'featured');
    }

    public function scopeWeekendDeals($query)
    {
        return $query->where('placement_type', 'weekend_deals');
    }

    public function scopeTopUnique($query)
    {
        return $query->where('placement_type', 'top_unique');
    }

    // Accessors
    public function getPlacementTypeNameAttribute()
    {
        return match($this->placement_type) {
            'featured' => 'Featured Listings',
            'weekend_deals' => 'Weekend Deals',
            'top_unique' => 'Top Unique Properties',
            default => ucfirst(str_replace('_', ' ', $this->placement_type))
        };
    }

    public function getPlacementTypeIconAttribute()
    {
        return match($this->placement_type) {
            'featured' => 'fa-star',
            'weekend_deals' => 'fa-calendar-week',
            'top_unique' => 'fa-gem',
            default => 'fa-tag'
        };
    }

    public function getPlacementTypeColorAttribute()
    {
        return match($this->placement_type) {
            'featured' => 'blue',
            'weekend_deals' => 'orange',
            'top_unique' => 'purple',
            default => 'gray'
        };
    }
}