<?php

namespace App\Http\Controllers;

use App\Models\Booking;
use App\Models\VendorAd;
use App\Models\Transaction;
use App\Services\Payment\PaystackService;
use App\Services\Payment\FlutterwaveService;
use App\Services\BookingService;
use App\Services\CommissionService;
use App\Events\PaymentVerified;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class WebhookController extends Controller
{
    public function paystack(Request $request, PaystackService $paystackService)
    {
        $signature = $request->header('X-Paystack-Signature');
        $payload = $request->all();
        
        if (!$paystackService->verifyWebhook($payload, $signature)) {
            Log::warning('Invalid Paystack webhook signature', ['payload' => $payload]);
            return response()->json(['error' => 'Invalid signature'], 400);
        }
        
        $event = $payload['event'];
        $data = $payload['data'];
        
        if ($event === 'charge.success') {
            return $this->handleSuccessfulPayment($data, 'paystack');
        }
        
        return response()->json(['status' => 'success']);
    }
    
    public function flutterwave(Request $request, FlutterwaveService $flutterwaveService)
    {
        $signature = $request->header('verif-hash');
        $payload = $request->all();
        
        if (!$flutterwaveService->verifyWebhook($payload, $signature)) {
            Log::warning('Invalid Flutterwave webhook signature', ['payload' => $payload]);
            return response()->json(['error' => 'Invalid signature'], 400);
        }
        
        $event = $payload['event'];
        $data = $payload['data'];
        
        if ($event === 'charge.completed' && $data['status'] === 'successful') {
            return $this->handleSuccessfulPayment($data, 'flutterwave');
        }
        
        return response()->json(['status' => 'success']);
    }
    
    protected function handleSuccessfulPayment(array $data, string $provider)
    {
        $reference = $provider === 'paystack' ? $data['reference'] : $data['tx_ref'];
        $amount = $provider === 'paystack' ? $data['amount'] / 100 : $data['amount'];
        
        // Check if it's a booking payment
        $booking = Booking::where('payment_reference', $reference)->first();
        
        if ($booking) {
            if ($booking->payment_status === 'paid') {
                return response()->json(['status' => 'already_processed']);
            }
            
            $booking->update([
                'payment_status' => 'paid',
                'payment_method' => $provider,
                'paid_at' => now(),
            ]);
            
            // Track discount deal usage if discount was applied
            if ($booking->discount_deal_id) {
                $discountService = app(\App\Services\DiscountDealService::class);
                $discountDeal = \App\Models\DiscountDeal::find($booking->discount_deal_id);
                
                if ($discountDeal) {
                    $discountService->applyDiscountDeal(
                        $discountDeal,
                        $booking->user,
                        $booking,
                        $booking->subtotal + $booking->caution_fee_amount,
                        $booking->discount_amount,
                        $booking->total_amount
                    );
                }
            }
            
            // Create transaction record
            Transaction::create([
                'user_id' => $booking->user_id,
                'vendor_id' => $booking->vendor_id,
                'booking_id' => $booking->id,
                'type' => 'booking_payment',
                'reference' => $reference,
                'amount' => $amount,
                'currency' => $data['currency'] ?? 'NGN',
                'status' => 'completed',
                'payment_method' => $provider,
                'payment_gateway_reference' => $data['id'] ?? null,
                'gateway_response' => $data,
                'description' => "Payment for booking #{$booking->id}",
            ]);
            
            // Credit vendor pending balance
            app(CommissionService::class)->creditVendorBalance($booking);
            
            // Cancel conflicting unpaid bookings if no units available
            $bookingService = app(BookingService::class);
            $cancelledCount = $bookingService->cancelConflictingUnpaidBookings($booking);
            
            if ($cancelledCount > 0) {
                Log::info('Auto-cancelled conflicting bookings', [
                    'paid_booking_id' => $booking->id,
                    'cancelled_count' => $cancelledCount,
                ]);
            }
            
            event(new PaymentVerified($booking));
            
            Log::info('Booking payment processed', [
                'booking_id' => $booking->id,
                'reference' => $reference,
                'amount' => $amount,
            ]);
            
            return response()->json(['status' => 'success', 'type' => 'booking']);
        }
        
        // Check if it's an ad payment
        $vendorAd = VendorAd::where('payment_reference', $reference)->first();
        
        if ($vendorAd) {
            if ($vendorAd->payment_status === 'paid') {
                return response()->json(['status' => 'already_processed']);
            }
            
            $vendorAd->update([
                'payment_status' => 'paid',
                'paid_at' => now(),
            ]);
            
            // Track promo code usage if promo code was applied
            if ($vendorAd->promo_code_id) {
                $promoCodeService = app(\App\Services\PromoCodeService::class);
                $promoCode = \App\Models\PromoCode::find($vendorAd->promo_code_id);
                
                if ($promoCode) {
                    $promoCodeService->applyPromoCode(
                        $promoCode,
                        $vendorAd->vendor,
                        $vendorAd,
                        $vendorAd->adPlan->price,
                        $vendorAd->discount_amount,
                        $vendorAd->final_amount
                    );
                }
            }
            
            Transaction::create([
                'vendor_id' => $vendorAd->vendor_id,
                'type' => 'ad_payment',
                'reference' => $reference,
                'amount' => $amount,
                'currency' => $data['currency'] ?? 'NGN',
                'status' => 'completed',
                'payment_method' => $provider,
                'payment_gateway_reference' => $data['id'] ?? null,
                'gateway_response' => $data,
                'description' => "Ad payment for property #{$vendorAd->property_id}",
            ]);
            
            Log::info('Ad payment processed', [
                'vendor_ad_id' => $vendorAd->id,
                'reference' => $reference,
                'amount' => $amount,
            ]);
            
            return response()->json(['status' => 'success', 'type' => 'ad']);
        }
        
        Log::warning('Payment reference not found', ['reference' => $reference]);
        
        return response()->json(['status' => 'reference_not_found'], 404);
    }
}