<?php

namespace App\Http\Controllers\Vendor;

use App\Http\Controllers\Controller;
use App\Models\Property;
use App\Models\PropertyRoom;
use App\Services\Maps\GoogleMapsService;
use Illuminate\Http\Request;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;

class PropertyController extends Controller
{
    use AuthorizesRequests;
    
    public function index()
    {
        $properties = auth()->user()
            ->vendor
            ->properties()
            ->withCount('bookings')
            ->latest()
            ->paginate(15);
        
        return view('vendor.properties.index', compact('properties'));
    }
    
    public function create()
    {
        return view('vendor.properties.create');
    }
    
    public function show(Property $property)
    {
        $this->authorize('view', $property);
        
        $property->load(['bookings' => function ($query) {
            $query->latest()->limit(10);
        }, 'rooms']);
        
        return view('vendor.properties.show', compact('property'));
    }
    
    public function store(Request $request, GoogleMapsService $mapsService)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'is_hotel' => 'boolean',
            'price_per_night' => 'required_if:is_hotel,false|nullable|numeric|min:0',
            'caution_fee' => 'required|numeric|min:0',
            'allow_parties' => 'boolean',
            'address' => 'required|string',
            'city' => 'required|string',
            'state' => 'required|string',
            'house_type' => 'required|string',
            'bedrooms' => 'required_if:is_hotel,false|nullable|integer|min:0',
            'beds' => 'required_if:is_hotel,false|nullable|integer|min:0',
            'bathrooms' => 'required_if:is_hotel,false|nullable|integer|min:0',
            'max_guests' => 'required_if:is_hotel,false|nullable|integer|min:1',
            'total_units' => 'required_if:is_hotel,false|nullable|integer|min:1',
            'check_in_from' => 'required|date_format:H:i',
            'check_in_until' => 'required|date_format:H:i',
            'check_out_time' => 'required|date_format:H:i',
            'highlighted_features' => 'nullable|array',
            'amenities' => 'nullable|array',
            'house_rules' => 'nullable|array',
            'photos' => 'nullable|array',
            'photos.*' => 'image|max:5120',
            'video' => 'nullable|file|mimes:mp4,mov,avi,wmv|max:51200', // 50MB max
            'location_lat' => 'nullable|numeric',
            'location_lng' => 'nullable|numeric',
            
            // Hotel rooms
            'rooms' => 'required_if:is_hotel,true|nullable|array|min:1',
            'rooms.*.room_type' => 'required_with:rooms|string|max:100',
            'rooms.*.description' => 'nullable|string',
            'rooms.*.price_per_night' => 'required_with:rooms|numeric|min:0',
            'rooms.*.max_guests' => 'required_with:rooms|integer|min:1',
            'rooms.*.total_units' => 'required_with:rooms|integer|min:1',
            'rooms.*.bedrooms' => 'required_with:rooms|integer|min:0',
            'rooms.*.beds' => 'required_with:rooms|integer|min:0',
            'rooms.*.bathrooms' => 'required_with:rooms|integer|min:0',
            'rooms.*.room_size_sqm' => 'nullable|numeric|min:0',
            'rooms.*.amenities' => 'nullable|array',
            'rooms.*.photos' => 'nullable|array',
            'rooms.*.photos.*' => 'image|max:5120',
        ]);
        
        // Geocode address
        $fullAddress = "{$validated['address']}, {$validated['city']}, {$validated['state']}, Nigeria";
        $coordinates = $mapsService->geocodeAddress($fullAddress);
        
        if ($coordinates) {
            $validated['location_lat'] = $coordinates['latitude'];
            $validated['location_lng'] = $coordinates['longitude'];
        }
        
        // Handle video upload
        if ($request->hasFile('video')) {
            $videoPath = $request->file('video')->store('properties/videos', 'public');
            $validated['video_url'] = $videoPath;
        }
        
        // Handle photo uploads
        if ($request->hasFile('photos')) {
            $photoPaths = [];
            $watermarkService = new \App\Services\ImageWatermarkService();
            
            foreach ($request->file('photos') as $photo) {
                $path = $photo->store('properties', 'public');
                $watermarkedPath = $watermarkService->addWatermark($path, 'center', 50);
                $photoPaths[] = $watermarkedPath;
            }
            $validated['photos'] = $photoPaths;
        }
        
        $validated['vendor_id'] = auth()->user()->vendor->id;
        $validated['status'] = 'pending';
        
        // For hotels, set basic property fields to null/default
        if ($validated['is_hotel'] ?? false) {
            $validated['bedrooms'] = 0;
            $validated['beds'] = 0;
            $validated['bathrooms'] = 0;
            $validated['max_guests'] = 0;
            $validated['total_units'] = 0;
            $validated['price_per_night'] = 0;
        }
        
        $property = Property::create($validated);
        
        // Handle hotel rooms
        if (($validated['is_hotel'] ?? false) && !empty($validated['rooms'])) {
            foreach ($validated['rooms'] as $roomData) {
                // Handle room photos
                if (!empty($roomData['photos'])) {
                    $roomPhotoPaths = [];
                    $watermarkService = new \App\Services\ImageWatermarkService();
                    
                    foreach ($roomData['photos'] as $photo) {
                        if ($photo instanceof \Illuminate\Http\UploadedFile) {
                            $path = $photo->store('properties/rooms', 'public');
                            $watermarkedPath = $watermarkService->addWatermark($path, 'center', 50);
                            $roomPhotoPaths[] = $watermarkedPath;
                        }
                    }
                    $roomData['photos'] = $roomPhotoPaths;
                }
                
                $property->rooms()->create($roomData);
            }
        }
        
        return redirect()->route('vendor.properties.show', $property)
            ->with('success', 'Property submitted successfully! It will be reviewed by our team.');
    }
    
    public function edit(Property $property)
    {
        $this->authorize('update', $property);
        
        $property->load('rooms');
        
        return view('vendor.properties.edit', compact('property'));
    }
    
    public function update(Request $request, Property $property)
    {
        $this->authorize('update', $property);
        
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'price_per_night' => 'required_if:is_hotel,false|nullable|numeric|min:0',
            'caution_fee' => 'required|numeric|min:0',
            'status' => 'required|in:active,inactive',
            'unavailable_dates' => 'nullable|array',
            'video' => 'nullable|file|mimes:mp4,mov,avi,wmv|max:51200',
            'delete_video' => 'nullable|boolean',
        ]);
        
        // Handle video upload
        if ($request->hasFile('video')) {
            // Delete old video
            if ($property->video_url) {
                \Storage::disk('public')->delete($property->video_url);
            }
            $videoPath = $request->file('video')->store('properties/videos', 'public');
            $validated['video_url'] = $videoPath;
        }
        
        // Handle video deletion
        if ($request->boolean('delete_video') && $property->video_url) {
            \Storage::disk('public')->delete($property->video_url);
            $validated['video_url'] = null;
        }
        
        $property->update($validated);
        
        return redirect()->route('vendor.properties.show', $property)
            ->with('success', 'Property updated successfully!');
    }
    
    public function destroy(Property $property)
    {
        $this->authorize('delete', $property);
        
        $activeBookings = $property->bookings()
            ->whereIn('booking_status', ['confirmed', 'pending'])
            ->where('payment_status', 'paid')
            ->where('end_date', '>=', now())
            ->count();
        
        if ($activeBookings > 0) {
            return back()->with('error', 'Cannot delete property with active bookings.');
        }
        
        // Delete property photos
        if ($property->photos && is_array($property->photos)) {
            foreach ($property->photos as $photo) {
                \Storage::disk('public')->delete($photo);
            }
        }
        
        // Delete video
        if ($property->video_url) {
            \Storage::disk('public')->delete($property->video_url);
        }
        
        // Delete room photos for hotels
        if ($property->is_hotel) {
            foreach ($property->rooms as $room) {
                if ($room->photos && is_array($room->photos)) {
                    foreach ($room->photos as $photo) {
                        \Storage::disk('public')->delete($photo);
                    }
                }
            }
        }
        
        $propertyTitle = $property->title;
        $property->delete();
        
        return redirect()->route('vendor.properties.index')
            ->with('success', "Property '{$propertyTitle}' has been deleted successfully.");
    }
}