<?php

namespace App\Http\Controllers\Vendor;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;

class ProfileController extends Controller
{
    public function edit()
    {
        $vendor = auth()->user()->vendor;
        
        if (!$vendor) {
            return redirect()->route('vendor.dashboard')
                ->with('error', 'Vendor profile not found. Please contact support.');
        }
        
        return view('vendor.profile.edit', compact('vendor'));
    }

    public function update(Request $request)
    {
        $vendor = auth()->user()->vendor;
        
        if (!$vendor) {
            return redirect()->route('vendor.dashboard')
                ->with('error', 'Vendor profile not found. Please contact support.');
        }

        // Check if this is a bank details update (no company_name field)
        if (!$request->has('company_name')) {
            // Bank details update only
            $validated = $request->validate([
                'bank_name' => 'nullable|string|max:255',
                'account_number' => 'nullable|string|max:20',
                'account_name' => 'nullable|string|max:255',
            ]);

            // Validate account name matches legal name or company name if KYC is approved
            if ($vendor->kyc_status === 'approved' && !empty($validated['account_name'])) {
                $accountName = strtolower(trim($validated['account_name']));
                $companyName = strtolower(trim($vendor->company_name));
                $legalName = strtolower(trim(auth()->user()->name));
                
                // Check if account name contains either company name or legal name
                $matchesCompany = str_contains($accountName, $companyName) || str_contains($companyName, $accountName);
                $matchesLegal = str_contains($accountName, $legalName) || str_contains($legalName, $accountName);
                
                if (!$matchesCompany && !$matchesLegal) {
                    return back()->withErrors([
                        'account_name' => 'Account name must match your legal name or business name on KYC documents.'
                    ])->withInput();
                }
            }

            $vendor->update($validated);

            return back()->with('success', 'Bank details updated successfully.');
        }

        // Full profile update
        $validated = $request->validate([
            'company_name' => 'required|string|max:255',
            'phone' => 'nullable|string|max:20',
            'whatsapp_number' => 'nullable|string|max:20',
            'bank_name' => 'nullable|string|max:255',
            'account_number' => 'nullable|string|max:20',
            'account_name' => 'nullable|string|max:255',
        ]);

        // Security: Prevent company name change after KYC approval
        if ($vendor->kyc_status === 'approved' && $validated['company_name'] !== $vendor->company_name) {
            return back()->withErrors([
                'company_name' => 'Company name cannot be changed after KYC approval for security reasons.'
            ])->withInput();
        }

        // Validate account name matches legal name or company name if KYC is approved
        if ($vendor->kyc_status === 'approved' && !empty($validated['account_name'])) {
            $accountName = strtolower(trim($validated['account_name']));
            $companyName = strtolower(trim($vendor->company_name));
            $legalName = strtolower(trim(auth()->user()->name));
            
            // Check if account name contains either company name or legal name
            $matchesCompany = str_contains($accountName, $companyName) || str_contains($companyName, $accountName);
            $matchesLegal = str_contains($accountName, $legalName) || str_contains($legalName, $accountName);
            
            if (!$matchesCompany && !$matchesLegal) {
                return back()->withErrors([
                    'account_name' => 'Account name must match your legal name or business name on KYC documents.'
                ])->withInput();
            }
        }

        $vendor->update($validated);

        return back()->with('success', 'Profile updated successfully.');
    }

    public function updatePassword(Request $request)
    {
        $validated = $request->validate([
            'current_password' => 'required',
            'password' => 'required|min:8|confirmed',
        ]);

        if (!Hash::check($validated['current_password'], auth()->user()->password)) {
            return back()->withErrors(['current_password' => 'Current password is incorrect.']);
        }

        auth()->user()->update([
            'password' => Hash::make($validated['password']),
        ]);

        return back()->with('success', 'Password updated successfully.');
    }

    public function updatePin(Request $request)
    {
        $vendor = auth()->user()->vendor;
        
        if (!$vendor) {
            return redirect()->route('vendor.dashboard')
                ->with('error', 'Vendor profile not found.');
        }

        // Validate PIN
        $validated = $request->validate([
            'current_pin' => $vendor->transaction_pin ? 'required|digits:4' : 'nullable',
            'transaction_pin' => 'required|digits:4|confirmed',
        ]);

        // If vendor has existing PIN, verify it
        if ($vendor->transaction_pin) {
            if (!Hash::check($validated['current_pin'], $vendor->transaction_pin)) {
                return back()->withErrors(['current_pin' => 'Current PIN is incorrect.']);
            }
        }

        // Update PIN
        $vendor->update([
            'transaction_pin' => Hash::make($validated['transaction_pin']),
            'pin_set_at' => now(),
        ]);

        return back()->with('success', 'Transaction PIN ' . ($vendor->wasChanged('transaction_pin') ? 'updated' : 'set') . ' successfully.');
    }
}
