<?php

namespace App\Http\Controllers\Vendor;

use App\Http\Controllers\Controller;
use App\Models\Car;
use App\Models\CarCategory;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class CarController extends Controller
{
    public function index()
    {
        $vendor = auth()->user()->vendor;
        
        // Check if vendor offers car rentals
        if (!$vendor->offersCarRentals()) {
            return redirect()->route('vendor.dashboard')
                ->with('error', 'You need to enable car rental service first.');
        }
        
        $cars = $vendor->cars()->with('category')->latest()->paginate(10);
        
        return view('vendor.cars.index', compact('cars', 'vendor'));
    }
    
    public function create()
    {
        $vendor = auth()->user()->vendor;
        
        // Check if vendor offers car rentals and KYC is approved
        if (!$vendor->offersCarRentals()) {
            return redirect()->route('vendor.dashboard')
                ->with('error', 'You need to enable car rental service first.');
        }
        
        if (!$vendor->isCarRentalKycApproved()) {
            return redirect()->route('vendor.kyc.show')
                ->with('error', 'Please complete your car rental KYC verification first.');
        }
        
        $categories = CarCategory::active()->ordered()->get();
        
        return view('vendor.cars.create', compact('categories'));
    }
    
    public function store(Request $request)
    {
        $vendor = auth()->user()->vendor;
        
        if (!$vendor->offersCarRentals() || !$vendor->isCarRentalKycApproved()) {
            return back()->with('error', 'You are not authorized to add cars.');
        }
        
        $validated = $request->validate([
            'car_category_id' => 'required|exists:car_categories,id',
            'brand' => 'required|string|max:100',
            'model' => 'required|string|max:100',
            'year' => 'required|integer|min:1900|max:' . (date('Y') + 1),
            'plate_number' => 'required|string|max:20|unique:cars,plate_number',
            'color' => 'required|string|max:50',
            'transmission' => 'required|in:automatic,manual',
            'fuel_type' => 'required|in:petrol,diesel,electric,hybrid',
            'seats' => 'required|integer|min:1|max:50',
            'doors' => 'required|integer|min:2|max:10',
            'luggage_capacity' => 'required|integer|min:0|max:20',
            'air_conditioning' => 'boolean',
            'features' => 'nullable|array',
            'state' => 'required|string|max:100',
            'city' => 'required|string|max:100',
            'pickup_location' => 'required|string|max:500',
            'price_per_day' => 'required|numeric|min:0',
            'price_per_week' => 'nullable|numeric|min:0',
            'price_per_month' => 'nullable|numeric|min:0',
            'security_deposit' => 'required|numeric|min:0',
            'minimum_rental_days' => 'required|integer|min:1',
            'maximum_rental_days' => 'nullable|integer|min:1',
            'insurance_included' => 'boolean',
            'insurance_cost_per_day' => 'nullable|numeric|min:0',
            'mileage_limit_per_day' => 'nullable|integer|min:0',
            'extra_mileage_cost' => 'nullable|numeric|min:0',
            'minimum_driver_age' => 'required|integer|min:18|max:100',
            'minimum_license_years' => 'required|integer|min:0|max:50',
            'international_license_accepted' => 'boolean',
            'description' => 'nullable|string',
            'photos' => 'required|array|min:1|max:10',
            'photos.*' => 'image|mimes:jpeg,jpg,png|max:5120',
            'cancellation_policy' => 'nullable|string',
            'rental_terms' => 'nullable|string',
        ]);
        
        // Handle photo uploads
        $photoPaths = [];
        if ($request->hasFile('photos')) {
            $watermarkService = new \App\Services\ImageWatermarkService();
            
            foreach ($request->file('photos') as $photo) {
                $path = $photo->store('cars', 'public');
                
                // Add watermark to the uploaded image (center position)
                $watermarkedPath = $watermarkService->addWatermark($path, 'center', 50);
                $photoPaths[] = $watermarkedPath;
            }
        }
        
        $validated['photos'] = $photoPaths;
        $validated['vendor_id'] = $vendor->id;
        $validated['air_conditioning'] = $request->has('air_conditioning');
        $validated['insurance_included'] = $request->has('insurance_included');
        $validated['international_license_accepted'] = $request->has('international_license_accepted');
        $validated['status'] = 'pending';
        $validated['is_available'] = true;
        
        Car::create($validated);
        
        return redirect()->route('vendor.cars.index')
            ->with('success', 'Car added successfully! It will be reviewed by admin before going live.');
    }
    
    public function edit(Car $car)
    {
        $vendor = auth()->user()->vendor;
        
        // Check ownership
        if ($car->vendor_id !== $vendor->id) {
            abort(403, 'Unauthorized action.');
        }
        
        $categories = CarCategory::active()->ordered()->get();
        
        return view('vendor.cars.edit', compact('car', 'categories'));
    }
    
    public function update(Request $request, Car $car)
    {
        $vendor = auth()->user()->vendor;
        
        // Check ownership
        if ($car->vendor_id !== $vendor->id) {
            abort(403, 'Unauthorized action.');
        }
        
        $validated = $request->validate([
            'car_category_id' => 'required|exists:car_categories,id',
            'brand' => 'required|string|max:100',
            'model' => 'required|string|max:100',
            'year' => 'required|integer|min:1900|max:' . (date('Y') + 1),
            'plate_number' => 'required|string|max:20|unique:cars,plate_number,' . $car->id,
            'color' => 'required|string|max:50',
            'transmission' => 'required|in:automatic,manual',
            'fuel_type' => 'required|in:petrol,diesel,electric,hybrid',
            'seats' => 'required|integer|min:1|max:50',
            'doors' => 'required|integer|min:2|max:10',
            'luggage_capacity' => 'required|integer|min:0|max:20',
            'air_conditioning' => 'boolean',
            'features' => 'nullable|array',
            'state' => 'required|string|max:100',
            'city' => 'required|string|max:100',
            'pickup_location' => 'required|string|max:500',
            'price_per_day' => 'required|numeric|min:0',
            'price_per_week' => 'nullable|numeric|min:0',
            'price_per_month' => 'nullable|numeric|min:0',
            'security_deposit' => 'required|numeric|min:0',
            'minimum_rental_days' => 'required|integer|min:1',
            'maximum_rental_days' => 'nullable|integer|min:1',
            'insurance_included' => 'boolean',
            'insurance_cost_per_day' => 'nullable|numeric|min:0',
            'mileage_limit_per_day' => 'nullable|integer|min:0',
            'extra_mileage_cost' => 'nullable|numeric|min:0',
            'minimum_driver_age' => 'required|integer|min:18|max:100',
            'minimum_license_years' => 'required|integer|min:0|max:50',
            'international_license_accepted' => 'boolean',
            'description' => 'nullable|string',
            'photos' => 'nullable|array|max:10',
            'photos.*' => 'image|mimes:jpeg,jpg,png|max:5120',
            'cancellation_policy' => 'nullable|string',
            'rental_terms' => 'nullable|string',
            'is_available' => 'boolean',
        ]);
        
        // Handle new photo uploads
        if ($request->hasFile('photos')) {
            $photoPaths = $car->photos ?? [];
            $watermarkService = new \App\Services\ImageWatermarkService();
            
            foreach ($request->file('photos') as $photo) {
                $path = $photo->store('cars', 'public');
                
                // Add watermark to the uploaded image (center position)
                $watermarkedPath = $watermarkService->addWatermark($path, 'center', 50);
                $photoPaths[] = $watermarkedPath;
            }
            
            $validated['photos'] = $photoPaths;
        }
        
        $validated['air_conditioning'] = $request->has('air_conditioning');
        $validated['insurance_included'] = $request->has('insurance_included');
        $validated['international_license_accepted'] = $request->has('international_license_accepted');
        $validated['is_available'] = $request->has('is_available');
        
        $car->update($validated);
        
        return redirect()->route('vendor.cars.index')
            ->with('success', 'Car updated successfully!');
    }
    
    public function destroy(Car $car)
    {
        $vendor = auth()->user()->vendor;
        
        // Check ownership
        if ($car->vendor_id !== $vendor->id) {
            abort(403, 'Unauthorized action.');
        }
        
        // Check if car has active bookings
        $hasActiveBookings = $car->bookings()
            ->whereIn('status', ['confirmed', 'active'])
            ->exists();
        
        if ($hasActiveBookings) {
            return back()->with('error', 'Cannot delete car with active bookings.');
        }
        
        // Delete photos
        if ($car->photos) {
            foreach ($car->photos as $photo) {
                Storage::disk('public')->delete($photo);
            }
        }
        
        $car->delete();
        
        return redirect()->route('vendor.cars.index')
            ->with('success', 'Car deleted successfully!');
    }
    
    public function deletePhoto(Car $car, $index)
    {
        $vendor = auth()->user()->vendor;
        
        // Check ownership
        if ($car->vendor_id !== $vendor->id) {
            abort(403, 'Unauthorized action.');
        }
        
        $photos = $car->photos ?? [];
        
        if (isset($photos[$index])) {
            // Delete the file
            Storage::disk('public')->delete($photos[$index]);
            
            // Remove from array
            unset($photos[$index]);
            
            // Reindex array
            $photos = array_values($photos);
            
            $car->update(['photos' => $photos]);
            
            return response()->json(['success' => true]);
        }
        
        return response()->json(['success' => false], 404);
    }
}
