<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Booking;
use App\Models\CarBooking;

class TransactionController extends Controller
{
    public function index(Request $request)
    {
        $user = auth()->user();
        $type = $request->get('type', 'all'); // all, bookings, refunds
        
        $transactions = collect();

        // Property Bookings
        if (in_array($type, ['all', 'bookings'])) {
            $propertyBookings = Booking::where('user_id', $user->id)
                ->where('payment_status', 'paid')
                ->with('property')
                ->get()
                ->map(function ($booking) {
                    return [
                        'id' => $booking->id,
                        'type' => 'booking',
                        'service' => 'property',
                        'description' => 'Property Booking - ' . $booking->property->title,
                        'reference' => $booking->id,
                        'amount' => $booking->total_amount,
                        'date' => $booking->paid_at,
                        'status' => 'paid',
                        'details' => [
                            'check_in' => $booking->start_date,
                            'check_out' => $booking->end_date,
                            'nights' => $booking->nights_count,
                        ]
                    ];
                });
            $transactions = $transactions->merge($propertyBookings);
        }

        // Car Bookings
        if (in_array($type, ['all', 'bookings'])) {
            $carBookings = CarBooking::where('user_id', $user->id)
                ->where('payment_status', 'paid')
                ->with('car')
                ->get()
                ->map(function ($booking) {
                    return [
                        'id' => $booking->id,
                        'type' => 'booking',
                        'service' => 'car',
                        'description' => 'Car Rental - ' . $booking->car->full_name,
                        'reference' => $booking->booking_reference,
                        'amount' => $booking->total_amount,
                        'date' => $booking->paid_at,
                        'status' => 'paid',
                        'details' => [
                            'pickup_date' => $booking->pickup_date,
                            'return_date' => $booking->return_date,
                            'days' => $booking->rental_days,
                        ]
                    ];
                });
            $transactions = $transactions->merge($carBookings);
        }

        // Caution Fee Refunds (Property)
        if (in_array($type, ['all', 'refunds'])) {
            $cautionRefunds = Booking::where('user_id', $user->id)
                ->whereIn('caution_refund_status', ['refunded', 'partially_deducted', 'deducted'])
                ->whereNotNull('caution_refund_amount')
                ->where('caution_refund_amount', '>', 0)
                ->with('property')
                ->get()
                ->map(function ($booking) {
                    return [
                        'id' => $booking->id,
                        'type' => 'refund',
                        'service' => 'property',
                        'description' => 'Caution Fee Refund - ' . $booking->property->title,
                        'reference' => $booking->id,
                        'amount' => $booking->caution_refund_amount,
                        'date' => $booking->caution_refunded_at,
                        'status' => $booking->refund_payout_status,
                        'details' => [
                            'original_amount' => $booking->caution_fee_amount,
                            'deduction' => $booking->caution_deduction_amount ?? 0,
                            'refund_status' => $booking->caution_refund_status,
                            'payout_status' => $booking->refund_payout_status,
                        ]
                    ];
                });
            $transactions = $transactions->merge($cautionRefunds);
        }

        // Security Deposit Refunds (Car)
        if (in_array($type, ['all', 'refunds'])) {
            $depositRefunds = CarBooking::where('user_id', $user->id)
                ->whereIn('deposit_status', ['refunded', 'partially_deducted', 'deducted'])
                ->whereNotNull('deposit_refund_amount')
                ->where('deposit_refund_amount', '>', 0)
                ->with('car')
                ->get()
                ->map(function ($booking) {
                    return [
                        'id' => $booking->id,
                        'type' => 'refund',
                        'service' => 'car',
                        'description' => 'Security Deposit Refund - ' . $booking->car->full_name,
                        'reference' => $booking->booking_reference,
                        'amount' => $booking->deposit_refund_amount,
                        'date' => $booking->deposit_refunded_at,
                        'status' => $booking->deposit_payout_status ?? 'pending',
                        'details' => [
                            'original_amount' => $booking->security_deposit,
                            'deduction' => $booking->deposit_deduction_amount ?? 0,
                            'refund_status' => $booking->deposit_status,
                        ]
                    ];
                });
            $transactions = $transactions->merge($depositRefunds);
        }

        // Sort by date
        $transactions = $transactions->sortByDesc('date')->values();

        return view('user.transactions.index', compact('transactions', 'type'));
    }
}
