<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use App\Models\Booking;
use App\Models\Review;
use Illuminate\Http\Request;

class ReviewController extends Controller
{
    public function create(Booking $booking)
    {
        // Check if user can review
        if ($booking->user_id !== auth()->id()) {
            abort(403);
        }

        if (!$booking->canReview()) {
            return back()->withErrors(['review' => 'You can only review completed bookings.']);
        }

        return view('user.reviews.create', compact('booking'));
    }

    public function store(Request $request, Booking $booking)
    {
        // Authorize
        if ($booking->user_id !== auth()->id() || !$booking->canReview()) {
            abort(403);
        }

        $validated = $request->validate([
            'rating' => 'required|integer|min:1|max:5',
            'comment' => 'nullable|string|max:1000',
            'cleanliness_rating' => 'nullable|integer|min:1|max:5',
            'accuracy_rating' => 'nullable|integer|min:1|max:5',
            'checkin_rating' => 'nullable|integer|min:1|max:5',
            'communication_rating' => 'nullable|integer|min:1|max:5',
            'location_rating' => 'nullable|integer|min:1|max:5',
            'value_rating' => 'nullable|integer|min:1|max:5',
        ]);

        Review::create([
            'user_id' => auth()->id(),
            'property_id' => $booking->property_id,
            'booking_id' => $booking->id,
            'rating' => $validated['rating'],
            'comment' => $validated['comment'],
            'cleanliness_rating' => $validated['cleanliness_rating'] ?? null,
            'accuracy_rating' => $validated['accuracy_rating'] ?? null,
            'checkin_rating' => $validated['checkin_rating'] ?? null,
            'communication_rating' => $validated['communication_rating'] ?? null,
            'location_rating' => $validated['location_rating'] ?? null,
            'value_rating' => $validated['value_rating'] ?? null,
        ]);

        return redirect()->route('bookings.show', $booking)
            ->with('success', 'Review submitted successfully!');
    }

    public function index()
    {
        $reviews = auth()->user()
            ->reviews()
            ->with(['property', 'booking'])
            ->latest()
            ->paginate(10);

        return view('user.reviews.index', compact('reviews'));
    }
}