<?php

namespace App\Http\Controllers;

use App\Models\Booking;
use App\Models\Property;
use App\Models\Review;
use Illuminate\Http\Request;

class ReviewController extends Controller
{
    public function create(Request $request)
    {
        $propertyId = $request->get('property');
        $bookingId = $request->get('booking');

        // Validate that the booking exists and belongs to the user
        $booking = Booking::where('id', $bookingId)
            ->where('user_id', auth()->id())
            ->where('property_id', $propertyId)
            ->where('booking_status', 'completed')
            ->firstOrFail();

        // Check if user already reviewed this booking
        $existingReview = Review::where('booking_id', $booking->id)
            ->where('user_id', auth()->id())
            ->first();

        if ($existingReview) {
            return redirect()->route('bookings.show', $booking)
                ->with('info', 'You have already reviewed this property.');
        }

        $property = $booking->property;

        return view('reviews.create', compact('booking', 'property'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'booking_id' => 'required|exists:bookings,id',
            'property_id' => 'required|exists:properties,id',
            'rating' => 'required|integer|min:1|max:5',
            'cleanliness_rating' => 'required|integer|min:1|max:5',
            'accuracy_rating' => 'required|integer|min:1|max:5',
            'communication_rating' => 'required|integer|min:1|max:5',
            'location_rating' => 'required|integer|min:1|max:5',
            'value_rating' => 'required|integer|min:1|max:5',
            'comment' => 'required|string|max:1000',
        ]);

        // Verify booking belongs to user and is completed
        $booking = Booking::where('id', $validated['booking_id'])
            ->where('user_id', auth()->id())
            ->where('property_id', $validated['property_id'])
            ->where('booking_status', 'completed')
            ->firstOrFail();

        // Check if already reviewed
        $existingReview = Review::where('booking_id', $booking->id)
            ->where('user_id', auth()->id())
            ->first();

        if ($existingReview) {
            return redirect()->route('bookings.show', $booking)
                ->with('info', 'You have already reviewed this property.');
        }

        // Create review
        Review::create([
            'booking_id' => $validated['booking_id'],
            'property_id' => $validated['property_id'],
            'user_id' => auth()->id(),
            'rating' => $validated['rating'],
            'cleanliness_rating' => $validated['cleanliness_rating'],
            'accuracy_rating' => $validated['accuracy_rating'],
            'communication_rating' => $validated['communication_rating'],
            'location_rating' => $validated['location_rating'],
            'value_rating' => $validated['value_rating'],
            'comment' => $validated['comment'],
        ]);

        return redirect()->route('bookings.show', $booking)
            ->with('success', 'Thank you for your review! It has been submitted successfully.');
    }
}
