<?php

// app/Http/Controllers/PropertyController.php

namespace App\Http\Controllers;

use App\Models\Property;
use App\Services\Maps\PlacesService;
use Illuminate\Http\Request;

class PropertyController extends Controller
{
    public function index(Request $request)
    {
        $query = Property::query()
            ->with(['vendor.user', 'reviews'])
            ->active();
        
        // State filter
        if ($request->filled('state')) {
            $query->where('state', $request->state);
        }
        
        // City filter
        if ($request->filled('city')) {
            $query->where('city', 'like', '%' . $request->city . '%');
        }
        
        // Guests filter
        if ($request->filled('guests')) {
            $query->where('max_guests', '>=', $request->guests);
        }
        
        // Bedrooms filter
        if ($request->filled('bedrooms')) {
            $query->where('bedrooms', '>=', $request->bedrooms);
        }
        
        // Price filters - handle min and max separately
        if ($request->filled('min_price')) {
            $query->where('price_per_night', '>=', $request->min_price);
        }
        
        if ($request->filled('max_price')) {
            $query->where('price_per_night', '<=', $request->max_price);
        }
        
        // Property type filter
        if ($request->filled('house_type')) {
            $query->where('house_type', $request->house_type);
        }
        
        // Party friendly filter
        if ($request->boolean('allow_parties')) {
            $query->where('allow_parties', true);
        }
        
        // Sorting
        $sortBy = $request->get('sort', 'featured');
        
        switch ($sortBy) {
            case 'price_low':
                $query->orderBy('price_per_night', 'asc');
                break;
            case 'price_high':
                $query->orderBy('price_per_night', 'desc');
                break;
            case 'newest':
                $query->orderBy('created_at', 'desc');
                break;
            case 'rating':
                $query->withAvg('reviews', 'rating')
                      ->orderBy('reviews_avg_rating', 'desc');
                break;
            default:
                // Featured properties first, then newest
                $query->orderByRaw('featured_until IS NOT NULL AND featured_until >= NOW() DESC');
                $query->orderBy('created_at', 'desc');
        }
        
        $properties = $query->paginate(20)->withQueryString();
        
        return view('properties.index', compact('properties'));
    }
    
    public function show(Property $property)
    {
        $property->load(['vendor.user', 'reviews.user']);
        
        // Calculate average rating
        $averageRating = $property->averageRating();
        
        // Get review counts by rating
        $ratingCounts = $property->reviews()
            ->where('visible', true)
            ->selectRaw('rating, COUNT(*) as count')
            ->groupBy('rating')
            ->pluck('count', 'rating');
        
        return view('properties.show', compact('property', 'averageRating', 'ratingCounts'));
    }
    
    public function showNearbyAttractions(Property $property, PlacesService $placesService)
    {
        // Check if user has a verified booking for this property
        $booking = $property->bookings()
            ->where('user_id', auth()->id())
            ->where('payment_status', 'paid')
            ->where('admin_verified', true)
            ->first();
        
        if (!$booking) {
            abort(403, 'You must have a verified booking to view this information.');
        }
        
        $attractions = $placesService->getNearbyAttractions($property);
        
        return view('properties.nearby-attractions', compact('property', 'booking', 'attractions'));
    }
}
