<?php

namespace App\Http\Controllers;

use App\Models\Property;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Session;

class GuestActivityController extends Controller
{
    /**
     * Track a search
     */
    public function trackSearch(Request $request)
    {
        $searches = Session::get('recent_searches', []);
        
        $search = [
            'location' => $request->input('location'),
            'state' => $request->input('state'),
            'city' => $request->input('city'),
            'check_in' => $request->input('check_in'),
            'check_out' => $request->input('check_out'),
            'guests' => $request->input('guests', 1),
            'timestamp' => now()->toDateTimeString(),
        ];
        
        // Remove duplicates and keep only last 5 searches
        $searches = array_filter($searches, function($s) use ($search) {
            return $s['location'] !== $search['location'] || 
                   $s['check_in'] !== $search['check_in'];
        });
        
        array_unshift($searches, $search);
        $searches = array_slice($searches, 0, 5);
        
        Session::put('recent_searches', $searches);
        
        return response()->json(['success' => true]);
    }
    
    /**
     * Track a viewed property
     */
    public function trackPropertyView(Request $request, Property $property)
    {
        $viewedProperties = Session::get('viewed_properties', []);
        
        // Remove if already exists
        $viewedProperties = array_filter($viewedProperties, function($id) use ($property) {
            return $id !== $property->id;
        });
        
        // Add to beginning
        array_unshift($viewedProperties, $property->id);
        
        // Keep only last 10
        $viewedProperties = array_slice($viewedProperties, 0, 10);
        
        Session::put('viewed_properties', $viewedProperties);
        
        return response()->json(['success' => true]);
    }
    
    /**
     * Get recent searches
     */
    public function getRecentSearches()
    {
        $searches = Session::get('recent_searches', []);
        return response()->json($searches);
    }
    
    /**
     * Get viewed properties
     */
    public function getViewedProperties()
    {
        $propertyIds = Session::get('viewed_properties', []);
        $properties = Property::whereIn('id', $propertyIds)
            ->with(['images', 'reviews'])
            ->get()
            ->sortBy(function($property) use ($propertyIds) {
                return array_search($property->id, $propertyIds);
            })
            ->values();
        
        return response()->json($properties);
    }
    
    /**
     * Clear recent activity
     */
    public function clearActivity(Request $request)
    {
        $type = $request->input('type', 'all');
        
        if ($type === 'searches' || $type === 'all') {
            Session::forget('recent_searches');
        }
        
        if ($type === 'properties' || $type === 'all') {
            Session::forget('viewed_properties');
        }
        
        return response()->json(['success' => true]);
    }
}
