<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Property;
use App\Models\Booking;
use App\Services\BookingService;
use Illuminate\Http\Request;

class BookingController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth:sanctum');
    }

    public function index()
    {
        $bookings = auth()->user()
            ->bookings()
            ->with(['property', 'vendor'])
            ->latest()
            ->paginate(20);

        return response()->json([
            'data' => $bookings->items(),
            'meta' => [
                'current_page' => $bookings->currentPage(),
                'last_page' => $bookings->lastPage(),
                'per_page' => $bookings->perPage(),
                'total' => $bookings->total(),
            ],
        ]);
    }

    public function store(Request $request, BookingService $bookingService)
    {
        $validated = $request->validate([
            'property_id' => 'required|exists:properties,id',
            'start_date' => 'required|date|after:today',
            'end_date' => 'required|date|after:start_date',
            'adults_count' => 'required|integer|min:1',
            'children_count' => 'required|integer|min:0',
            'trip_type' => 'required|in:normal,party',
        ]);

        $property = Property::findOrFail($validated['property_id']);

        // Check availability
        if (!$property->isAvailable($validated['start_date'], $validated['end_date'])) {
            return response()->json([
                'message' => 'Property is not available for selected dates.',
                'errors' => ['dates' => ['Property is not available for selected dates.']],
            ], 422);
        }

        $booking = $bookingService->createBooking(auth()->user(), $property, $validated);

        return response()->json([
            'message' => 'Booking created successfully.',
            'data' => $booking->load(['property', 'vendor']),
        ], 201);
    }
}