<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Vendor;
use App\Models\User;
use Illuminate\Http\Request;

class VendorController extends Controller
{
    public function index(Request $request)
    {
        $query = Vendor::with('user')->withCount('properties');
        
        if ($request->filled('verified')) {
            $query->where('verified', $request->boolean('verified'));
        }
        
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('company_name', 'like', "%{$search}%")
                    ->orWhereHas('user', function ($userQuery) use ($search) {
                        $userQuery->where('name', 'like', "%{$search}%")
                            ->orWhere('email', 'like', "%{$search}%");
                    });
            });
        }
        
        $vendors = $query->latest()->paginate(20);
        
        return view('admin.vendors.index', compact('vendors'));
    }
    
    public function show(Vendor $vendor)
    {
        $vendor->load([
            'user',
            'properties',
            'bookings' => fn($q) => $q->latest()->limit(10),
            'payouts' => fn($q) => $q->latest()->limit(10),
        ]);
        
        return view('admin.vendors.show', compact('vendor'));
    }
    
    public function verify(Vendor $vendor)
    {
        $vendor->update(['verified' => true]);
        
        return back()->with('success', 'Vendor verified successfully.');
    }
    
    public function suspend(Request $request, Vendor $vendor)
    {
        $validated = $request->validate([
            'reason' => 'required|string|max:500',
        ]);
        
        $vendor->update(['verified' => false]);
        
        // Deactivate all properties
        $vendor->properties()->update(['status' => 'inactive']);
        
        return back()->with('success', 'Vendor suspended.');
    }

    public function resetPin(Vendor $vendor)
    {
        $vendor->update([
            'transaction_pin' => null,
            'pin_set_at' => null,
        ]);
        
        return back()->with('success', 'Transaction PIN reset successfully. Vendor will need to set a new PIN.');
    }
}
