<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Booking;
use App\Models\CarBooking;

class TransactionController extends Controller
{
    public function index(Request $request)
    {
        $type = $request->get('type', 'all'); // all, bookings, refunds
        $service = $request->get('service', 'all'); // all, property, car
        
        $transactions = collect();

        // Property Bookings
        if (in_array($type, ['all', 'bookings']) && in_array($service, ['all', 'property'])) {
            $propertyBookings = Booking::where('payment_status', 'paid')
                ->with(['property', 'user'])
                ->get()
                ->map(function ($booking) {
                    return [
                        'id' => $booking->id,
                        'type' => 'booking',
                        'service' => 'property',
                        'user' => $booking->user->name,
                        'description' => 'Property Booking - ' . $booking->property->title,
                        'reference' => $booking->id,
                        'amount' => $booking->total_amount,
                        'date' => $booking->paid_at,
                        'status' => 'paid',
                    ];
                });
            $transactions = $transactions->merge($propertyBookings);
        }

        // Car Bookings
        if (in_array($type, ['all', 'bookings']) && in_array($service, ['all', 'car'])) {
            $carBookings = CarBooking::where('payment_status', 'paid')
                ->with(['car', 'user'])
                ->get()
                ->map(function ($booking) {
                    return [
                        'id' => $booking->id,
                        'type' => 'booking',
                        'service' => 'car',
                        'user' => $booking->user->name,
                        'description' => 'Car Rental - ' . $booking->car->full_name,
                        'reference' => $booking->booking_reference,
                        'amount' => $booking->total_amount,
                        'date' => $booking->paid_at,
                        'status' => 'paid',
                    ];
                });
            $transactions = $transactions->merge($carBookings);
        }

        // Caution Fee Refunds (Property)
        if (in_array($type, ['all', 'refunds']) && in_array($service, ['all', 'property'])) {
            $cautionRefunds = Booking::whereIn('caution_refund_status', ['refunded', 'partially_deducted', 'deducted'])
                ->whereNotNull('caution_refund_amount')
                ->where('caution_refund_amount', '>', 0)
                ->with(['property', 'user'])
                ->get()
                ->map(function ($booking) {
                    return [
                        'id' => $booking->id,
                        'type' => 'refund',
                        'service' => 'property',
                        'user' => $booking->user->name,
                        'description' => 'Caution Fee Refund - ' . $booking->property->title,
                        'reference' => $booking->id,
                        'amount' => $booking->caution_refund_amount,
                        'date' => $booking->caution_refunded_at,
                        'status' => $booking->refund_payout_status ?? 'pending',
                        'payout_status' => $booking->refund_payout_status ?? 'pending',
                    ];
                });
            $transactions = $transactions->merge($cautionRefunds);
        }

        // Security Deposit Refunds (Car) - FIXED HERE
        if (in_array($type, ['all', 'refunds']) && in_array($service, ['all', 'car'])) {
            $depositRefunds = CarBooking::whereIn('deposit_status', ['refunded', 'partially_deducted', 'deducted'])
                ->whereNotNull('deposit_refund_amount')
                ->where('deposit_refund_amount', '>', 0)
                ->with(['car', 'user'])
                ->get()
                ->map(function ($booking) {
                    return [
                        'id' => $booking->id,
                        'type' => 'refund',
                        'service' => 'car',
                        'user' => $booking->user->name,
                        'description' => 'Security Deposit Refund - ' . $booking->car->full_name,
                        'reference' => $booking->booking_reference,
                        'amount' => $booking->deposit_refund_amount,
                        'date' => $booking->deposit_refunded_at,
                        'status' => $booking->refund_payout_status ?? 'pending', // FIXED: was deposit_payout_status
                        'payout_status' => $booking->refund_payout_status ?? 'pending', // FIXED: was deposit_payout_status
                    ];
                });
            $transactions = $transactions->merge($depositRefunds);
        }

        // Sort by date
        $transactions = $transactions->sortByDesc('date')->values();

        return view('admin.transactions.index', compact('transactions', 'type', 'service'));
    }

    public function payouts(Request $request)
    {
        $status = $request->get('status', 'all'); // all, pending, requested, processing, paid
        $service = $request->get('service', 'all'); // all, property, car
        
        $payouts = collect();

        // Property Caution Fee Refunds
        if (in_array($service, ['all', 'property'])) {
            $propertyRefunds = Booking::whereIn('caution_refund_status', ['refunded', 'partially_deducted', 'deducted'])
                ->whereNotNull('caution_refund_amount')
                ->where('caution_refund_amount', '>', 0)
                ->when($status !== 'all', function($q) use ($status) {
                    $q->where('refund_payout_status', $status);
                })
                ->with(['property', 'user'])
                ->get()
                ->map(function ($booking) {
                    return [
                        'id' => $booking->id,
                        'type' => 'property',
                        'service' => 'property',
                        'user' => $booking->user,
                        'property_name' => $booking->property->title,
                        'description' => 'Caution Fee Refund - ' . $booking->property->title,
                        'reference' => $booking->payment_reference,
                        'amount' => $booking->caution_refund_amount,
                        'requested_at' => $booking->refund_requested_at,
                        'paid_at' => $booking->refund_paid_at,
                        'status' => $booking->refund_payout_status ?? 'pending',
                        'payment_method' => $booking->refund_payment_method,
                        'payment_details' => $booking->refund_payment_details,
                        'booking' => $booking,
                    ];
                });
            $payouts = $payouts->merge($propertyRefunds);
        }

        // Car Security Deposit Refunds - ADDED
        if (in_array($service, ['all', 'car'])) {
            $carRefunds = CarBooking::whereIn('deposit_status', ['refunded', 'partially_deducted', 'deducted'])
                ->whereNotNull('deposit_refund_amount')
                ->where('deposit_refund_amount', '>', 0)
                ->when($status !== 'all', function($q) use ($status) {
                    $q->where('refund_payout_status', $status);
                })
                ->with(['car', 'user'])
                ->get()
                ->map(function ($booking) {
                    return [
                        'id' => $booking->id,
                        'type' => 'car',
                        'service' => 'car',
                        'user' => $booking->user,
                        'property_name' => $booking->car->full_name,
                        'description' => 'Security Deposit Refund - ' . $booking->car->full_name,
                        'reference' => $booking->booking_reference,
                        'amount' => $booking->deposit_refund_amount,
                        'requested_at' => $booking->refund_requested_at,
                        'paid_at' => $booking->refund_paid_at,
                        'status' => $booking->refund_payout_status ?? 'pending',
                        'payment_method' => $booking->refund_payment_method,
                        'payment_details' => $booking->refund_payment_details,
                        'booking' => $booking,
                    ];
                });
            $payouts = $payouts->merge($carRefunds);
        }

        // Sort by requested date (nulls last)
        $payouts = $payouts->sortByDesc(function($payout) {
            return $payout['requested_at'] ?? now()->subYears(100);
        })->values();

        // Calculate stats
        $stats = [
            'total' => $payouts->count(),
            'pending' => $payouts->where('status', 'pending')->count(),
            'requested' => $payouts->where('status', 'requested')->count(),
            'processing' => $payouts->where('status', 'processing')->count(),
            'paid' => $payouts->where('status', 'paid')->count(),
            'total_amount' => $payouts->sum('amount'),
            'pending_amount' => $payouts->whereIn('status', ['pending', 'requested', 'processing'])->sum('amount'),
        ];

        return view('admin.transactions.payouts', compact('payouts', 'stats', 'status', 'service'));
    }
}