<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\PopularDestination;
use App\Models\Property;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class PopularDestinationController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $destinations = PopularDestination::ordered()->paginate(20);

        return view('admin.destinations.index', compact('destinations'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        // Get unique states from properties
        $states = Property::select('state')
            ->distinct()
            ->orderBy('state')
            ->pluck('state');

        return view('admin.destinations.create', compact('states'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'type' => 'required|in:city,state',
            'state' => 'required_if:type,city|nullable|string|max:255',
            'image' => 'required|image|mimes:jpeg,jpg,png,webp|max:5120',
            'description' => 'nullable|string|max:500',
            'display_order' => 'nullable|integer|min:0',
            'active' => 'boolean',
        ]);

        // Handle image upload
        if ($request->hasFile('image')) {
            $validated['image'] = $request->file('image')->store('destinations', 'public');
        }

        $validated['active'] = $request->has('active');
        $validated['display_order'] = $validated['display_order'] ?? 0;

        PopularDestination::create($validated);

        return redirect()->route('admin.destinations.index')
            ->with('success', 'Popular destination created successfully!');
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(PopularDestination $destination)
    {
        // Get unique states from properties
        $states = Property::select('state')
            ->distinct()
            ->orderBy('state')
            ->pluck('state');

        return view('admin.destinations.edit', compact('destination', 'states'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, PopularDestination $destination)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'type' => 'required|in:city,state',
            'state' => 'required_if:type,city|nullable|string|max:255',
            'image' => 'nullable|image|mimes:jpeg,jpg,png,webp|max:5120',
            'description' => 'nullable|string|max:500',
            'display_order' => 'nullable|integer|min:0',
            'active' => 'boolean',
        ]);

        // Handle image upload
        if ($request->hasFile('image')) {
            // Delete old image
            if ($destination->image) {
                Storage::disk('public')->delete($destination->image);
            }
            $validated['image'] = $request->file('image')->store('destinations', 'public');
        }

        $validated['active'] = $request->has('active');
        $validated['display_order'] = $validated['display_order'] ?? 0;

        $destination->update($validated);

        return redirect()->route('admin.destinations.index')
            ->with('success', 'Popular destination updated successfully!');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(PopularDestination $destination)
    {
        // Delete image
        if ($destination->image) {
            Storage::disk('public')->delete($destination->image);
        }

        $destination->delete();

        return redirect()->route('admin.destinations.index')
            ->with('success', 'Popular destination deleted successfully!');
    }

    /**
     * Get property count for destination
     */
    private function getPropertyCount(PopularDestination $destination)
    {
        if ($destination->type === 'state') {
            return Property::active()->where('state', $destination->name)->count();
        } else {
            return Property::active()
                ->where('city', $destination->name)
                ->where('state', $destination->state)
                ->count();
        }
    }
}
