<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Vendor;
use App\Services\BusinessVerificationService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class KycApprovalController extends Controller
{
    protected $verificationService;

    public function __construct(BusinessVerificationService $verificationService)
    {
        $this->verificationService = $verificationService;
    }

    public function index(Request $request)
    {
        $status = $request->get('status', 'pending');
        
        $vendors = Vendor::with('user')
            ->where('kyc_status', $status)
            ->whereNotNull('kyc_submitted_at')
            ->latest('kyc_submitted_at')
            ->paginate(20);
        
        return view('admin.kyc.index', compact('vendors', 'status'));
    }
    
    /**
     * Verify business registration number
     */
    public function verifyBusiness(Request $request, Vendor $vendor)
    {
        try {
            $type = $request->get('type', 'property'); // 'property' or 'car_rental'
            
            if ($type === 'car_rental') {
                $registrationNumber = $vendor->car_rental_business_registration;
                $businessName = $vendor->car_rental_business_name;
                
                if (!$registrationNumber) {
                    return back()->with('error', 'No car rental registration number found for verification.');
                }
                
                if (!$businessName) {
                    return back()->with('error', 'No car rental business name found for verification.');
                }
            } else {
                // For property vendors
                return back()->with('error', 'Property business verification is not yet implemented. Please verify car rental businesses only.');
            }

            // Log the verification attempt
            Log::info('Starting business verification', [
                'vendor_id' => $vendor->id,
                'type' => $type,
                'registration_number' => $registrationNumber,
                'business_name' => $businessName,
            ]);

            // Call verification API
            $result = $this->verificationService->verifyBusiness($registrationNumber);

            // Log the API response
            Log::info('Business verification API response', [
                'vendor_id' => $vendor->id,
                'result' => $result,
            ]);

            // Handle API failure
            if (!$result['success']) {
                $errorMessage = $result['message'] ?? 'Unknown error occurred';
                Log::error('Business verification API failed', [
                    'vendor_id' => $vendor->id,
                    'error' => $errorMessage,
                ]);
                return back()->with('error', 'Business verification failed: ' . $errorMessage);
            }

            // Get the verification status
            $verificationStatus = $result['status'] ?? 'unknown';

            // Handle 'not_found' status
            if ($verificationStatus === 'not_found') {
                $reason = $result['data']['reason'] ?? 'Business not found in CAC registry';
                
                Log::warning('Business not found in registry', [
                    'vendor_id' => $vendor->id,
                    'registration_number' => $registrationNumber,
                    'reason' => $reason,
                ]);

                // Store the failed verification attempt
                if ($type === 'car_rental') {
                    $vendor->update([
                        'car_rental_business_verified' => false,
                        'car_rental_verification_data' => $result['data'],
                        'car_rental_business_verified_at' => now(),
                    ]);
                }

                return back()->with('warning', 
                    'Business not found in CAC registry. ' . 
                    'Registration number may be invalid or the business may not be registered. ' .
                    'Reason: ' . $reason
                );
            }

            // Handle 'found' status
            if ($verificationStatus === 'found') {
                $verificationData = $result['data'];
                
                // Check if company is active
                $companyStatus = $verificationData['companyStatus'] ?? 'Unknown';
                if ($companyStatus !== 'ACTIVE') {
                    Log::warning('Business found but not active', [
                        'vendor_id' => $vendor->id,
                        'company_status' => $companyStatus,
                    ]);
                }

                // Validate business details match
                $validation = $this->verificationService->validateBusinessMatch(
                    $verificationData,
                    ['business_name' => $businessName]
                );

                // Store verification data
                if ($type === 'car_rental') {
                    $vendor->update([
                        'car_rental_business_verified' => $validation['is_valid'] && $companyStatus === 'ACTIVE',
                        'car_rental_verification_data' => $verificationData,
                        'car_rental_business_verified_at' => now(),
                    ]);
                } else {
                    $vendor->update([
                        'business_verified' => $validation['is_valid'] && $companyStatus === 'ACTIVE',
                        'business_verification_data' => $verificationData,
                        'business_verified_at' => now(),
                    ]);
                }

                // Prepare success/warning messages
                $messages = [];
                
                if ($companyStatus === 'ACTIVE') {
                    $messages[] = 'Company Status: ACTIVE ✓';
                } else {
                    $messages[] = "Company Status: {$companyStatus} ⚠";
                }

                if (!empty($validation['matches'])) {
                    $messages = array_merge($messages, $validation['matches']);
                }

                if (!empty($validation['mismatches'])) {
                    $messages = array_merge($messages, $validation['mismatches']);
                }

                // Determine overall success
                if ($validation['is_valid'] && $companyStatus === 'ACTIVE') {
                    return back()->with('success', 
                        'Business verified successfully! ' . 
                        implode(' | ', $messages)
                    );
                } else {
                    return back()->with('warning', 
                        'Business found but there are some concerns: ' . 
                        implode(' | ', $messages)
                    );
                }
            }

            // Handle any other unexpected status
            Log::error('Unexpected verification status', [
                'vendor_id' => $vendor->id,
                'status' => $verificationStatus,
                'full_result' => $result,
            ]);

            return back()->with('error', 
                "Unexpected verification status: '{$verificationStatus}'. " .
                "Please contact technical support or try again later."
            );

        } catch (\Exception $e) {
            Log::error('Business verification exception', [
                'vendor_id' => $vendor->id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);

            return back()->with('error', 
                'An error occurred during business verification: ' . $e->getMessage()
            );
        }
    }
    
    public function approve(Vendor $vendor)
    {
        // Check if car rental business is verified (if applicable)
        if ($vendor->offersCarRentals() && !$vendor->car_rental_business_verified) {
            return back()->with('error', 'Please verify the car rental business registration before approving KYC.');
        }

        $vendor->update([
            'kyc_status' => 'approved',
            'kyc_approved_at' => now(),
            'kyc_rejection_reason' => null,
        ]);
        
        // Log activity
        \App\Helpers\ActivityLogger::log(
            'approve_kyc',
            "Approved KYC for {$vendor->company_name}",
            $vendor
        );
        
        // TODO: Send notification to vendor
        
        return back()->with('success', 'KYC approved successfully.');
    }
    
    public function reject(Request $request, Vendor $vendor)
    {
        $validated = $request->validate([
            'kyc_rejection_reason' => 'required|string|max:1000',
        ]);
        
        $vendor->update([
            'kyc_status' => 'rejected',
            'kyc_rejection_reason' => $validated['kyc_rejection_reason'],
        ]);
        
        // Log activity
        \App\Helpers\ActivityLogger::log(
            'reject_kyc',
            "Rejected KYC for {$vendor->company_name}",
            $vendor,
            ['reason' => $validated['kyc_rejection_reason']]
        );
        
        // TODO: Send notification to vendor
        
        return back()->with('success', 'KYC rejected.');
    }
}