<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\DiscountDeal;
use Illuminate\Http\Request;

class DiscountDealController extends Controller
{
    public function index()
    {
        $discountDeals = DiscountDeal::with('creator')
            ->withCount('usages')
            ->latest()
            ->paginate(20);
        
        return view('admin.discount-deals.index', compact('discountDeals'));
    }
    
    public function create()
    {
        return view('admin.discount-deals.create');
    }
    
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'code' => 'required|string|max:50|unique:discount_deals,code|alpha_dash',
            'description' => 'nullable|string',
            'discount_type' => 'required|in:percentage,fixed',
            'discount_value' => 'required|numeric|min:0',
            'max_discount_amount' => 'nullable|numeric|min:0',
            'min_booking_amount' => 'required|numeric|min:0',
            'usage_limit' => 'nullable|integer|min:1',
            'usage_per_user' => 'required|integer|min:1',
            'valid_from' => 'nullable|date',
            'valid_until' => 'nullable|date|after:valid_from',
            'is_active' => 'boolean',
            'is_featured' => 'boolean',
            'show_on_homepage' => 'boolean',
            'banner_title' => 'nullable|string|max:255',
            'banner_description' => 'nullable|string',
            'banner_button_text' => 'nullable|string|max:100',
            'banner_button_link' => 'nullable|string|max:255',
            'banner_bg_from' => 'nullable|string|max:50',
            'banner_bg_to' => 'nullable|string|max:50',
            'banner_icon' => 'nullable|string|max:50',
        ]);
        
        // Convert code to uppercase
        $validated['code'] = strtoupper($validated['code']);
        $validated['created_by'] = auth()->id();
        
        // Validate percentage value
        if ($validated['discount_type'] === 'percentage' && $validated['discount_value'] > 100) {
            return back()->withErrors(['discount_value' => 'Percentage cannot exceed 100%'])->withInput();
        }
        
        DiscountDeal::create($validated);
        
        return redirect()->route('admin.discount-deals.index')
            ->with('success', 'Discount deal created successfully!');
    }
    
    public function edit(DiscountDeal $discountDeal)
    {
        return view('admin.discount-deals.edit', compact('discountDeal'));
    }
    
    public function update(Request $request, DiscountDeal $discountDeal)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'code' => 'required|string|max:50|alpha_dash|unique:discount_deals,code,' . $discountDeal->id,
            'description' => 'nullable|string',
            'discount_type' => 'required|in:percentage,fixed',
            'discount_value' => 'required|numeric|min:0',
            'max_discount_amount' => 'nullable|numeric|min:0',
            'min_booking_amount' => 'required|numeric|min:0',
            'usage_limit' => 'nullable|integer|min:1',
            'usage_per_user' => 'required|integer|min:1',
            'valid_from' => 'nullable|date',
            'valid_until' => 'nullable|date|after:valid_from',
            'is_active' => 'boolean',
            'is_featured' => 'boolean',
            'show_on_homepage' => 'boolean',
            'banner_title' => 'nullable|string|max:255',
            'banner_description' => 'nullable|string',
            'banner_button_text' => 'nullable|string|max:100',
            'banner_button_link' => 'nullable|string|max:255',
            'banner_bg_from' => 'nullable|string|max:50',
            'banner_bg_to' => 'nullable|string|max:50',
            'banner_icon' => 'nullable|string|max:50',
        ]);
        
        // Convert code to uppercase
        $validated['code'] = strtoupper($validated['code']);
        
        // Validate percentage value
        if ($validated['discount_type'] === 'percentage' && $validated['discount_value'] > 100) {
            return back()->withErrors(['discount_value' => 'Percentage cannot exceed 100%'])->withInput();
        }
        
        $discountDeal->update($validated);
        
        return redirect()->route('admin.discount-deals.index')
            ->with('success', 'Discount deal updated successfully!');
    }
    
    public function destroy(DiscountDeal $discountDeal)
    {
        // Check if discount deal has been used
        if ($discountDeal->usage_count > 0) {
            return back()->with('error', 'Cannot delete discount deal that has been used. Deactivate it instead.');
        }
        
        $discountDeal->delete();
        
        return redirect()->route('admin.discount-deals.index')
            ->with('success', 'Discount deal deleted successfully!');
    }
    
    public function toggle(DiscountDeal $discountDeal)
    {
        $discountDeal->update(['is_active' => !$discountDeal->is_active]);
        
        $status = $discountDeal->is_active ? 'activated' : 'deactivated';
        
        return back()->with('success', "Discount deal {$status} successfully!");
    }
    
    public function toggleFeatured(DiscountDeal $discountDeal)
    {
        $discountDeal->update(['is_featured' => !$discountDeal->is_featured]);
        
        $status = $discountDeal->is_featured ? 'featured' : 'unfeatured';
        
        return back()->with('success', "Discount deal {$status} successfully!");
    }
}
