<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\CarBooking;
use App\Models\CarReview;
use App\Models\CarDepositLog;
use Illuminate\Http\Request;

class CarBookingController extends Controller
{
    public function index(Request $request)
    {
        $query = CarBooking::with(['car', 'user', 'vendor']);

        // Filter by status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Filter by payment status
        if ($request->filled('payment_status')) {
            $query->where('payment_status', $request->payment_status);
        }

        // Search
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('booking_reference', 'like', "%{$search}%")
                  ->orWhere('driver_name', 'like', "%{$search}%")
                  ->orWhere('driver_phone', 'like', "%{$search}%");
            });
        }

        $bookings = $query->latest()->paginate(20)->withQueryString();

        $stats = [
            'total' => CarBooking::count(),
            'pending' => CarBooking::where('status', 'pending')->count(),
            'confirmed' => CarBooking::where('status', 'confirmed')->count(),
            'active' => CarBooking::where('status', 'active')->count(),
            'completed' => CarBooking::where('status', 'completed')->count(),
            'cancelled' => CarBooking::where('status', 'cancelled')->count(),
        ];

        return view('admin.car-bookings.index', compact('bookings', 'stats'));
    }

    public function show(CarBooking $booking)
    {
        $booking->load(['car.category', 'user', 'vendor.user', 'review']);

        return view('admin.car-bookings.show', compact('booking'));
    }

    public function updateStatus(Request $request, CarBooking $booking)
    {
        $validated = $request->validate([
            'status' => 'required|in:pending,confirmed,active,completed,cancelled',
        ]);

        $booking->update(['status' => $validated['status']]);

        return back()->with('success', 'Booking status updated successfully!');
    }

    public function verifyPayment(CarBooking $booking)
    {
        $booking->update([
            'payment_status' => 'paid',
            'payment_verified_at' => now(),
        ]);

        // Credit vendor with rental amount after commission
        $vendor = $booking->vendor;
        $vendorPayout = $booking->vendor_payout_amount;

        // Add to vendor's pending balance
        $vendor->increment('pending_balance', $vendorPayout);
        $vendor->increment('total_earned', $vendorPayout);

        return back()->with('success', 'Payment verified successfully! ₦' . number_format($vendorPayout) . ' credited to vendor.');
    }

    public function resolveDeposit(Request $request, CarBooking $booking)
    {
        $validated = $request->validate([
            'decision' => 'required|in:approve_deduction,partial_deduction,reject_deduction',
            'approved_amount' => 'required_if:decision,partial_deduction|nullable|numeric|min:0|max:' . $booking->security_deposit,
            'admin_notes' => 'required|string|max:1000',
        ]);

        $vendor = $booking->vendor;

        switch ($validated['decision']) {
            case 'approve_deduction':
                // Approve full deduction - release to vendor
                $deductionAmount = $booking->deposit_deduction_amount;
                
                // Validate that vendor actually requested a deduction amount
                if (!$deductionAmount || $deductionAmount <= 0) {
                    return back()->withErrors(['error' => 'Vendor did not specify a deduction amount. Please ask vendor to resubmit the damage report.']);
                }
                
                $refundAmount = $booking->security_deposit - $deductionAmount;

                // Credit vendor with deduction amount
                $vendor->increment('pending_balance', $deductionAmount);
                $vendor->increment('total_earned', $deductionAmount);

                $booking->update([
                    'status' => 'completed',
                    'deposit_status' => 'deducted',
                    'deposit_refund_amount' => $refundAmount,
                    'deposit_refunded_at' => now(),
                    'admin_notes' => $validated['admin_notes'],
                ]);

                // Log the decision
                CarDepositLog::create([
                    'booking_id' => $booking->id,
                    'actor_id' => auth()->id(),
                    'actor_role' => 'admin',
                    'action' => 'approved',
                    'previous_status' => 'dispute_pending',
                    'new_status' => 'deducted',
                    'amount' => $deductionAmount,
                    'reason' => $validated['admin_notes'],
                ]);

                $message = "Damage claim approved. ₦" . number_format($deductionAmount) . " released to vendor. ₦" . number_format($refundAmount) . " will be refunded to customer.";
                break;

            case 'partial_deduction':
                // Approve partial deduction
                $deductionAmount = $validated['approved_amount'];
                $refundAmount = $booking->security_deposit - $deductionAmount;

                // Credit vendor with approved deduction amount
                $vendor->increment('pending_balance', $deductionAmount);
                $vendor->increment('total_earned', $deductionAmount);

                $booking->update([
                    'status' => 'completed',
                    'deposit_status' => 'partially_deducted',
                    'deposit_deduction_amount' => $deductionAmount,
                    'deposit_refund_amount' => $refundAmount,
                    'deposit_refunded_at' => now(),
                    'admin_notes' => $validated['admin_notes'],
                ]);

                // Log the decision
                CarDepositLog::create([
                    'booking_id' => $booking->id,
                    'actor_id' => auth()->id(),
                    'actor_role' => 'admin',
                    'action' => 'approved',
                    'previous_status' => 'dispute_pending',
                    'new_status' => 'partially_deducted',
                    'amount' => $deductionAmount,
                    'reason' => $validated['admin_notes'],
                ]);

                $message = "Partial deduction approved. ₦" . number_format($deductionAmount) . " released to vendor. ₦" . number_format($refundAmount) . " will be refunded to customer.";
                break;

            case 'reject_deduction':
                // Reject deduction - full refund to customer
                $booking->update([
                    'status' => 'completed',
                    'deposit_status' => 'refunded',
                    'deposit_refund_amount' => $booking->security_deposit,
                    'deposit_refunded_at' => now(),
                    'admin_notes' => $validated['admin_notes'],
                ]);

                // Log the decision
                CarDepositLog::create([
                    'booking_id' => $booking->id,
                    'actor_id' => auth()->id(),
                    'actor_role' => 'admin',
                    'action' => 'rejected',
                    'previous_status' => 'dispute_pending',
                    'new_status' => 'refunded',
                    'amount' => 0,
                    'reason' => $validated['admin_notes'],
                ]);

                $message = "Damage claim rejected. Full deposit of ₦" . number_format($booking->security_deposit) . " will be refunded to customer.";
                break;
        }

        return back()->with('success', $message);
    }

    public function processRefund(Request $request, CarBooking $booking)
    {
        $validated = $request->validate([
            'payment_method' => 'required|in:bank_transfer,cash,cheque,mobile_money',
            'payment_details' => 'required|string|max:1000',
        ]);

        // Check if refund was requested
        if ($booking->refund_payout_status !== 'requested') {
            return back()->withErrors(['error' => 'No refund request found for this booking.']);
        }

        // Check if there's an amount to refund
        if (!$booking->deposit_refund_amount || $booking->deposit_refund_amount <= 0) {
            return back()->withErrors(['error' => 'No refund amount available.']);
        }

        $booking->update([
            'refund_payout_status' => 'paid',
            'refund_paid_at' => now(),
            'refund_payment_method' => $validated['payment_method'],
            'refund_payment_details' => $validated['payment_details'],
        ]);

        // Log the refund payment
        CarDepositLog::create([
            'booking_id' => $booking->id,
            'actor_id' => auth()->id(),
            'actor_role' => 'admin',
            'action' => 'refund_paid',
            'previous_status' => 'requested',
            'new_status' => 'paid',
            'amount' => $booking->deposit_refund_amount,
            'reason' => 'Admin processed refund payment: ' . $validated['payment_details'],
        ]);

        return back()->with('success', 'Refund marked as paid successfully! User will be notified.');
    }

    public function approveReview(CarReview $review)
    {
        $review->update(['is_approved' => true]);

        return back()->with('success', 'Review approved successfully!');
    }

    public function rejectReview(CarReview $review)
    {
        $review->delete();

        return back()->with('success', 'Review rejected and deleted.');
    }
}