<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Booking;
use App\Services\BookingService;
use App\Models\CautionRefundLog;
use Illuminate\Http\Request;

class BookingController extends Controller
{
    public function index(Request $request)
    {
        $query = Booking::with(['user', 'property', 'vendor']);
        
        if ($request->filled('status')) {
            $query->where('booking_status', $request->status);
        }
        
        if ($request->filled('payment_status')) {
            $query->where('payment_status', $request->payment_status);
        }
        
        if ($request->filled('search')) {
            $search = $request->search;
            $query->whereHas('user', function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                    ->orWhere('email', 'like', "%{$search}%");
            });
        }
        
        $bookings = $query->latest()->paginate(20);
        
        return view('admin.bookings.index', compact('bookings'));
    }
    
    public function show(Booking $booking)
    {
        $booking->load([
            'user',
            'property.vendor.user',
            'verifiedBy',
            'cautionDecidedBy',
            'cautionRefundLogs.actor',
            'activities.user',
        ]);
        
        return view('admin.bookings.show', compact('booking'));
    }
    
    public function verifyPayment(Booking $booking, BookingService $bookingService)
    {
        if ($booking->payment_status !== 'paid') {
            return back()->withErrors(['error' => 'Payment has not been completed yet.']);
        }
        
        if ($booking->admin_verified) {
            return back()->withErrors(['error' => 'Booking already verified.']);
        }
        
        $bookingService->verifyPayment($booking, auth()->user());
        
        // Log activity
        \App\Helpers\ActivityLogger::log(
            'verify_payment',
            'Verified payment for Booking #' . $booking->id,
            $booking,
            ['booking_id' => $booking->id, 'property' => $booking->property->title]
        );
        
        return back()->with('success', 'Booking verified! User can now access property details.');
    }
    
    public function overrideCautionRefund(Request $request, Booking $booking)
    {
        $validated = $request->validate([
            'decision' => 'required|in:approved,rejected,refunded',
            'reason' => 'required|string|max:500',
        ]);
        
        $previousStatus = $booking->caution_refund_status;
        $newStatus = $validated['decision'];
        
        $booking->update([
            'caution_refund_status' => $newStatus,
            'caution_refund_decided_at' => now(),
            'caution_decided_by' => auth()->id(),
            'caution_refund_reason' => $validated['reason'],
        ]);
        
        // Log the override
        CautionRefundLog::create([
            'booking_id' => $booking->id,
            'actor_id' => auth()->id(),
            'actor_role' => auth()->user()->getRoleNames()->first(),
            'action' => 'overridden',
            'previous_status' => $previousStatus,
            'new_status' => $newStatus,
            'reason' => $validated['reason'],
            'amount' => $booking->caution_fee_amount,
        ]);
        
        // Log activity
        \App\Helpers\ActivityLogger::log(
            'override_caution_refund',
            "Overrode caution refund decision for Booking #{$booking->id} from {$previousStatus} to {$newStatus}",
            $booking,
            [
                'previous_status' => $previousStatus,
                'new_status' => $newStatus,
                'amount' => $booking->caution_fee_amount,
            ]
        );
        
        return back()->with('success', 'Caution refund decision overridden.');
    }
    
    public function cancel(Request $request, Booking $booking)
    {
        $validated = $request->validate([
            'reason' => 'required|string|max:500',
        ]);
        
        $booking->update([
            'booking_status' => 'cancelled',
            'cancelled_at' => now(),
            'cancellation_reason' => $validated['reason'],
        ]);
        
        // Log activity
        \App\Helpers\ActivityLogger::log(
            'cancel_booking',
            "Cancelled Booking #{$booking->id}",
            $booking,
            ['reason' => $validated['reason']]
        );
        
        return back()->with('success', 'Booking cancelled.');
    }

    public function resolveCaution(Request $request, Booking $booking)
    {
        $maxAmount = $booking->caution_fee_amount ?? $booking->caution_fee ?? 0;
        
        $validated = $request->validate([
            'decision' => 'required|in:approve_deduction,partial_deduction,reject_deduction',
            'approved_amount' => 'required_if:decision,partial_deduction|nullable|numeric|min:0|max:' . $maxAmount,
            'admin_notes' => 'required|string|max:1000',
        ]);

        $vendor = $booking->vendor;

        switch ($validated['decision']) {
            case 'approve_deduction':
                // Approve full deduction - release to vendor
                $deductionAmount = $booking->caution_deduction_amount;
                
                // Validate that vendor actually requested a deduction amount
                if (!$deductionAmount || $deductionAmount <= 0) {
                    return back()->withErrors(['error' => 'Vendor did not specify a deduction amount. Please ask vendor to resubmit the damage report.']);
                }
                
                $cautionFee = $booking->caution_fee_amount ?? $booking->caution_fee ?? 0;
                $refundAmount = $cautionFee - $deductionAmount;

                // Credit vendor with deduction amount
                $vendor->increment('pending_balance', $deductionAmount);
                $vendor->increment('total_earned', $deductionAmount);

                $booking->update([
                    'caution_refund_status' => 'deducted',
                    'caution_refund_amount' => $refundAmount,
                    'caution_refunded_at' => now(),
                    'caution_decided_by_id' => auth()->id(),
                    'admin_notes' => $validated['admin_notes'],
                ]);

                // Log the decision
                CautionRefundLog::create([
                    'booking_id' => $booking->id,
                    'actor_id' => auth()->id(),
                    'actor_role' => 'admin',
                    'action' => 'approved',
                    'previous_status' => 'pending_vendor',
                    'new_status' => 'deducted',
                    'amount' => $deductionAmount,
                    'reason' => $validated['admin_notes'],
                ]);

                $message = "Damage claim approved. ₦" . number_format($deductionAmount) . " released to vendor. ₦" . number_format($refundAmount) . " will be refunded to customer.";
                break;

            case 'partial_deduction':
                // Approve partial deduction
                $deductionAmount = $validated['approved_amount'];
                $cautionFee = $booking->caution_fee_amount ?? $booking->caution_fee ?? 0;
                $refundAmount = $cautionFee - $deductionAmount;

                // Credit vendor with approved deduction amount
                $vendor->increment('pending_balance', $deductionAmount);
                $vendor->increment('total_earned', $deductionAmount);

                $booking->update([
                    'caution_refund_status' => 'partially_deducted',
                    'caution_deduction_amount' => $deductionAmount,
                    'caution_refund_amount' => $refundAmount,
                    'caution_refunded_at' => now(),
                    'caution_decided_by_id' => auth()->id(),
                    'admin_notes' => $validated['admin_notes'],
                ]);

                // Log the decision
                CautionRefundLog::create([
                    'booking_id' => $booking->id,
                    'actor_id' => auth()->id(),
                    'actor_role' => 'admin',
                    'action' => 'approved',
                    'previous_status' => 'pending_vendor',
                    'new_status' => 'partially_deducted',
                    'amount' => $deductionAmount,
                    'reason' => $validated['admin_notes'],
                ]);

                $message = "Partial deduction approved. ₦" . number_format($deductionAmount) . " released to vendor. ₦" . number_format($refundAmount) . " will be refunded to customer.";
                break;

            case 'reject_deduction':
                // Reject deduction - full refund to customer
                $cautionFee = $booking->caution_fee_amount ?? $booking->caution_fee ?? 0;
                $booking->update([
                    'caution_refund_status' => 'refunded',
                    'caution_refund_amount' => $cautionFee,
                    'caution_refunded_at' => now(),
                    'caution_decided_by_id' => auth()->id(),
                    'admin_notes' => $validated['admin_notes'],
                ]);

                // Log the decision
                CautionRefundLog::create([
                    'booking_id' => $booking->id,
                    'actor_id' => auth()->id(),
                    'actor_role' => 'admin',
                    'action' => 'rejected',
                    'previous_status' => 'pending_vendor',
                    'new_status' => 'refunded',
                    'amount' => 0,
                    'reason' => $validated['admin_notes'],
                ]);

                $message = "Damage claim rejected. Full caution fee of ₦" . number_format($cautionFee) . " will be refunded to customer.";
                break;
        }

        return back()->with('success', $message);
    }

    public function processRefund(Request $request, Booking $booking)
    {
        $validated = $request->validate([
            'payment_method' => 'required|in:bank_transfer,cash,cheque,mobile_money',
            'payment_details' => 'required|string|max:1000',
        ]);

        // Check if refund was requested
        if ($booking->refund_payout_status !== 'requested') {
            return back()->withErrors(['error' => 'No refund request found for this booking.']);
        }

        // Check if there's an amount to refund
        if (!$booking->caution_refund_amount || $booking->caution_refund_amount <= 0) {
            return back()->withErrors(['error' => 'No refund amount available.']);
        }

        $booking->update([
            'refund_payout_status' => 'paid',
            'refund_paid_at' => now(),
            'refund_payment_method' => $validated['payment_method'],
            'refund_payment_details' => $validated['payment_details'],
        ]);

        // Log activity
        \App\Helpers\ActivityLogger::log(
            'refund_paid',
            "Admin processed refund payout for Booking #{$booking->id}",
            $booking,
            [
                'admin_id' => auth()->id(),
                'admin_name' => auth()->user()->name,
                'refund_amount' => $booking->caution_refund_amount,
                'payment_method' => $validated['payment_method'],
                'paid_at' => now()->format('Y-m-d H:i:s')
            ]
        );

        return back()->with('success', 'Refund marked as paid successfully! User will be notified.');
    }
}