<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Booking;
use Illuminate\Http\Request;

class AllUsersController extends Controller
{
    public function index(Request $request)
    {
        $query = User::with(['vendor', 'bookings']);
        
        // Search filter
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%")
                  ->orWhere('phone', 'like', "%{$search}%");
            });
        }
        
        // Role filter
        if ($request->filled('role')) {
            if ($request->role === 'vendor') {
                $query->role('vendor');
            } elseif ($request->role === 'customer') {
                $query->whereDoesntHave('vendor')
                      ->whereDoesntHave('roles', function($q) {
                          $q->whereIn('name', ['owner', 'manager', 'moderator']);
                      });
            } elseif ($request->role === 'admin') {
                $query->role(['owner', 'manager', 'moderator']);
            }
        }
        
        // Status filter
        if ($request->filled('status')) {
            if ($request->status === 'verified') {
                $query->whereNotNull('email_verified_at');
            } elseif ($request->status === 'unverified') {
                $query->whereNull('email_verified_at');
            }
        }
        
        $users = $query->latest()->paginate(20);
        
        // Statistics
        $stats = [
            'total_users' => User::count(),
            'total_customers' => User::whereDoesntHave('vendor')
                ->whereDoesntHave('roles', function($q) {
                    $q->whereIn('name', ['owner', 'manager', 'moderator']);
                })->count(),
            'total_vendors' => User::role('vendor')->count(),
            'total_admins' => User::role(['owner', 'manager', 'moderator'])->count(),
            'verified_users' => User::whereNotNull('email_verified_at')->count(),
            'total_bookings' => Booking::count(),
            'total_booking_amount' => Booking::where('payment_status', 'paid')->sum('total_amount'),
            'total_revenue' => Booking::where('payment_status', 'paid')->sum('commission_amount'),
        ];
        
        return view('admin.all-users.index', compact('users', 'stats'));
    }
    
    public function show(User $user)
    {
        $user->load(['vendor', 'bookings.property', 'reviews']);
        
        $userStats = [
            'total_bookings' => $user->bookings()->count(),
            'completed_bookings' => $user->bookings()->where('booking_status', 'completed')->count(),
            'total_spent' => $user->bookings()->where('payment_status', 'paid')->sum('total_amount'),
            'total_reviews' => $user->reviews()->count(),
            'average_rating' => $user->reviews()->avg('rating'),
        ];
        
        return view('admin.all-users.show', compact('user', 'userStats'));
    }
}
