<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\VendorAd;
use Illuminate\Http\Request;

class AdApprovalController extends Controller
{
    public function index()
    {
        $pendingAds = VendorAd::with(['vendor.user', 'property', 'adPlan'])
            ->pendingApproval()
            ->latest()
            ->paginate(20);
        
        $activeAds = VendorAd::with(['vendor.user', 'property', 'adPlan'])
            ->active()
            ->latest()
            ->paginate(20);
        
        return view('admin.ads.index', compact('pendingAds', 'activeAds'));
    }
    
    public function approve(VendorAd $vendorAd)
    {
        if ($vendorAd->payment_status !== 'paid') {
            return back()->withErrors(['error' => 'Ad payment must be completed first.']);
        }
        
        $startDate = now();
        $endDate = $startDate->copy()->addDays($vendorAd->adPlan->duration_days);
        
        $vendorAd->update([
            'approved_by_admin' => true,
            'approved_by' => auth()->id(),
            'approved_at' => now(),
            'start_date' => $startDate,
            'end_date' => $endDate,
            'active' => true,
        ]);
        
        // Update property based on placement type
        $updateData = [];
        
        switch ($vendorAd->placement_type) {
            case 'featured':
                $updateData['featured_until'] = $endDate;
                $updateData['ad_plan_id'] = $vendorAd->ad_plan_id;
                break;
            case 'weekend_deals':
                $updateData['weekend_deals_until'] = $endDate;
                $updateData['weekend_deals_plan_id'] = $vendorAd->ad_plan_id;
                break;
            case 'top_unique':
                $updateData['top_unique_until'] = $endDate;
                $updateData['top_unique_plan_id'] = $vendorAd->ad_plan_id;
                break;
        }
        
        $vendorAd->property->update($updateData);
        
        return back()->with('success', 'Ad approved and activated.');
    }
    
    public function deactivate(VendorAd $vendorAd)
    {
        $vendorAd->update(['active' => false]);
        
        // Update property based on placement type
        $updateData = [];
        
        switch ($vendorAd->placement_type) {
            case 'featured':
                $updateData['featured_until'] = null;
                $updateData['ad_plan_id'] = null;
                break;
            case 'weekend_deals':
                $updateData['weekend_deals_until'] = null;
                $updateData['weekend_deals_plan_id'] = null;
                break;
            case 'top_unique':
                $updateData['top_unique_until'] = null;
                $updateData['top_unique_plan_id'] = null;
                break;
        }
        
        $vendorAd->property->update($updateData);
        
        return back()->with('success', 'Ad deactivated.');
    }
    
    public function extend(Request $request, VendorAd $vendorAd)
    {
        $validated = $request->validate([
            'days' => 'required|integer|min:1|max:90',
        ]);
        
        $newEndDate = $vendorAd->end_date->addDays($validated['days']);
        
        $vendorAd->update([
            'end_date' => $newEndDate,
        ]);
        
        // Update property based on placement type
        $updateData = [];
        
        switch ($vendorAd->placement_type) {
            case 'featured':
                $updateData['featured_until'] = $newEndDate;
                break;
            case 'weekend_deals':
                $updateData['weekend_deals_until'] = $newEndDate;
                break;
            case 'top_unique':
                $updateData['top_unique_until'] = $newEndDate;
                break;
        }
        
        $vendorAd->property->update($updateData);
        
        return back()->with('success', "Ad extended by {$validated['days']} days.");
    }
}