# ShortStayNG Security Guide

## 🔒 Security Implementation Checklist

### ✅ IMPLEMENTED SECURITY MEASURES

#### 1. **Authentication & Authorization**
- ✅ Laravel Sanctum for API authentication
- ✅ Spatie Laravel Permission for role-based access control (RBAC)
- ✅ Multi-role system: Owner, Manager, Moderator, Vendor, User
- ✅ Email verification required for users
- ✅ KYC verification required for vendors
- ✅ Transaction PIN for vendor payouts (4-digit, hashed)

#### 2. **Payment Security**
- ✅ Paystack & Flutterwave integration (PCI-DSS compliant)
- ✅ Webhook signature verification for payment callbacks
- ✅ Transaction reference validation
- ✅ Duplicate payment prevention
- ✅ Commission calculation server-side (never trust client)
- ✅ Vendor balance tracking with audit trail
- ✅ Transaction PIN required for payout requests

#### 3. **Data Protection**
- ✅ Password hashing with bcrypt (12 rounds)
- ✅ Transaction PIN hashing
- ✅ Sensitive data encrypted in database
- ✅ CSRF protection on all forms
- ✅ SQL injection prevention (Eloquent ORM)
- ✅ XSS protection (Blade escaping)

#### 4. **File Upload Security**
- ✅ File type validation (images only)
- ✅ File size limits (2MB for logos, 5MB for property images)
- ✅ Secure storage (Laravel Storage)
- ✅ No direct file execution

#### 5. **Session Security**
- ✅ Session stored in database
- ✅ Session lifetime: 120 minutes
- ✅ Secure session cookies
- ✅ Session regeneration on login

#### 6. **API Security**
- ✅ Rate limiting on API endpoints
- ✅ Token-based authentication
- ✅ Input validation on all requests
- ✅ JSON responses only

#### 7. **Audit & Logging**
- ✅ Activity logging for admin actions
- ✅ Payment transaction logs
- ✅ Booking modification logs
- ✅ Caution refund decision logs
- ✅ IP address tracking
- ✅ User agent tracking

---

## 🚨 CRITICAL SECURITY ACTIONS REQUIRED

### **PRODUCTION DEPLOYMENT CHECKLIST**

#### 1. **Environment Configuration** (CRITICAL)
```bash
# .env file - MUST UPDATE BEFORE PRODUCTION
APP_ENV=production
APP_DEBUG=false  # NEVER true in production
APP_KEY=base64:GENERATE_NEW_KEY_WITH_php_artisan_key:generate

# Use strong, unique passwords
DB_PASSWORD=STRONG_RANDOM_PASSWORD_HERE

# Secure session settings
SESSION_ENCRYPT=true
SESSION_SECURE_COOKIE=true
SESSION_HTTP_ONLY=true
SESSION_SAME_SITE=strict

# Payment webhook secrets - GET FROM PROVIDERS
PAYSTACK_WEBHOOK_SECRET=your_actual_webhook_secret
FLUTTERWAVE_WEBHOOK_SECRET=your_actual_webhook_secret
```

#### 2. **SSL/HTTPS** (MANDATORY)
```bash
# Force HTTPS in production
# Add to .env
APP_URL=https://yourdomain.com
FORCE_HTTPS=true

# Update config/app.php
'url' => env('APP_URL', 'https://yourdomain.com'),
```

#### 3. **Database Security**
```sql
-- Create dedicated database user with limited privileges
CREATE USER 'shortstay_app'@'localhost' IDENTIFIED BY 'STRONG_PASSWORD';
GRANT SELECT, INSERT, UPDATE, DELETE ON shortstayng.* TO 'shortstay_app'@'localhost';
FLUSH PRIVILEGES;

-- Never use root user in production
-- Never expose database port publicly
```

#### 4. **File Permissions**
```bash
# Set correct permissions
chmod -R 755 /path/to/shortstayng
chmod -R 775 storage bootstrap/cache
chown -R www-data:www-data /path/to/shortstayng

# Protect sensitive files
chmod 600 .env
chmod 600 config/database.php
```

#### 5. **Rate Limiting** (ADD TO ROUTES)
```php
// routes/web.php
Route::middleware(['throttle:60,1'])->group(function () {
    // Login attempts: 5 per minute
    Route::post('/login', [AuthController::class, 'login'])
        ->middleware('throttle:5,1');
    
    // Registration: 3 per hour
    Route::post('/register', [AuthController::class, 'register'])
        ->middleware('throttle:3,60');
    
    // Password reset: 3 per hour
    Route::post('/password/email', [PasswordResetController::class, 'sendResetLink'])
        ->middleware('throttle:3,60');
});

// API routes
Route::middleware(['throttle:api'])->group(function () {
    // API endpoints
});
```

#### 6. **Security Headers** (ADD TO MIDDLEWARE)
Create: `app/Http/Middleware/SecurityHeaders.php`
```php
<?php

namespace App\Http\Middleware;

use Closure;

class SecurityHeaders
{
    public function handle($request, Closure $next)
    {
        $response = $next($request);
        
        $response->headers->set('X-Content-Type-Options', 'nosniff');
        $response->headers->set('X-Frame-Options', 'SAMEORIGIN');
        $response->headers->set('X-XSS-Protection', '1; mode=block');
        $response->headers->set('Referrer-Policy', 'strict-origin-when-cross-origin');
        $response->headers->set('Permissions-Policy', 'geolocation=(), microphone=(), camera=()');
        
        if (config('app.env') === 'production') {
            $response->headers->set('Strict-Transport-Security', 'max-age=31536000; includeSubDomains');
        }
        
        return $response;
    }
}
```

#### 7. **Input Validation** (ALREADY IMPLEMENTED)
- ✅ All form requests validated
- ✅ Type casting enforced
- ✅ SQL injection prevented by Eloquent
- ✅ XSS prevented by Blade

#### 8. **Payment Webhook Verification** (ALREADY IMPLEMENTED)
```php
// WebhookController.php - Already secure
- Signature verification
- Duplicate prevention
- Amount validation
- Reference validation
```

#### 9. **Backup Strategy** (IMPLEMENT)
```bash
# Daily database backups
0 2 * * * /usr/bin/mysqldump -u backup_user -p'PASSWORD' shortstayng > /backups/db_$(date +\%Y\%m\%d).sql

# Weekly full backups
0 3 * * 0 tar -czf /backups/full_$(date +\%Y\%m\%d).tar.gz /path/to/shortstayng

# Encrypt backups
gpg --encrypt --recipient admin@shortstayng.com /backups/db_$(date +\%Y\%m\%d).sql

# Store backups off-site (AWS S3, Google Cloud, etc.)
```

#### 10. **Monitoring & Alerts**
```bash
# Install Laravel Telescope for debugging (ONLY IN STAGING)
composer require laravel/telescope --dev

# Production monitoring
- Set up error tracking (Sentry, Bugsnag)
- Monitor failed login attempts
- Alert on suspicious activities
- Track payment failures
```

---

## 🔐 ADDITIONAL SECURITY RECOMMENDATIONS

### **1. Two-Factor Authentication (2FA)**
Consider implementing 2FA for:
- Admin users (Owner, Manager, Moderator)
- Vendor accounts with high transaction volume
- Users before high-value bookings

### **2. IP Whitelisting for Admin**
```php
// Restrict admin access to specific IPs
Route::middleware(['admin.ip.whitelist'])->group(function () {
    // Admin routes
});
```

### **3. Fraud Detection**
- Monitor unusual booking patterns
- Flag rapid-fire bookings from same IP
- Alert on large payout requests
- Track failed payment attempts

### **4. Regular Security Audits**
```bash
# Run security checks
composer audit

# Update dependencies regularly
composer update --with-all-dependencies

# Check for known vulnerabilities
php artisan security:check
```

### **5. Vendor Verification**
- ✅ KYC documents required
- ✅ Admin approval required
- Consider: Bank account verification
- Consider: Phone number verification (OTP)

### **6. Payment Reconciliation**
- Daily reconciliation of payments
- Match Paystack/Flutterwave transactions with database
- Alert on discrepancies
- Audit vendor balances weekly

### **7. Data Retention Policy**
```php
// Auto-delete old data
- Cancelled bookings > 1 year
- Failed payments > 6 months
- Expired discount codes
- Old activity logs > 2 years
```

---

## 🚫 SECURITY DON'TS

### **NEVER DO THESE:**
1. ❌ Store payment card details (use Paystack/Flutterwave)
2. ❌ Store passwords in plain text
3. ❌ Trust client-side calculations (prices, commissions)
4. ❌ Expose API keys in frontend code
5. ❌ Use `APP_DEBUG=true` in production
6. ❌ Commit `.env` file to Git
7. ❌ Use default admin credentials
8. ❌ Allow file uploads without validation
9. ❌ Expose database errors to users
10. ❌ Use `SELECT *` with user input

---

## 📋 COMPLIANCE CHECKLIST

### **NDPR (Nigeria Data Protection Regulation)**
- ✅ Privacy policy page
- ✅ Terms of service
- ✅ User consent for data collection
- ✅ Right to data deletion
- ✅ Data encryption
- ✅ Secure data storage

### **PCI-DSS (Payment Card Industry)**
- ✅ No card data stored
- ✅ Use certified payment processors
- ✅ Encrypted transmission (HTTPS)
- ✅ Access control
- ✅ Regular security testing

---

## 🆘 INCIDENT RESPONSE PLAN

### **If Security Breach Detected:**

1. **Immediate Actions**
   - Take affected systems offline
   - Change all passwords and API keys
   - Revoke compromised tokens
   - Enable maintenance mode

2. **Investigation**
   - Review access logs
   - Identify breach point
   - Assess data exposure
   - Document findings

3. **Notification**
   - Notify affected users (if data exposed)
   - Report to NITDA (if required by NDPR)
   - Contact payment providers
   - Inform stakeholders

4. **Recovery**
   - Patch vulnerabilities
   - Restore from clean backups
   - Reset all credentials
   - Re-enable systems

5. **Post-Incident**
   - Conduct security audit
   - Update security measures
   - Train team on lessons learned
   - Implement additional controls

---

## 📞 SECURITY CONTACTS

**Security Issues:**
- Email: security@shortstayng.com
- Emergency: [Phone number]

**Payment Issues:**
- Paystack Support: support@paystack.com
- Flutterwave Support: support@flutterwave.com

---

## 📅 SECURITY MAINTENANCE SCHEDULE

### **Daily**
- Monitor error logs
- Check failed login attempts
- Review payment transactions

### **Weekly**
- Review activity logs
- Check vendor balances
- Update dependencies

### **Monthly**
- Security audit
- Backup verification
- Access review

### **Quarterly**
- Penetration testing
- Code review
- Policy updates

---

## ✅ DEPLOYMENT CHECKLIST

Before going live, ensure:

- [ ] APP_DEBUG=false
- [ ] APP_ENV=production
- [ ] HTTPS enabled
- [ ] SSL certificate installed
- [ ] Webhook secrets configured
- [ ] Database user has limited privileges
- [ ] File permissions set correctly
- [ ] Security headers middleware active
- [ ] Rate limiting enabled
- [ ] Backups configured
- [ ] Monitoring tools installed
- [ ] Error tracking configured
- [ ] Privacy policy published
- [ ] Terms of service published
- [ ] Security contact published

---

**Last Updated:** October 13, 2025
**Version:** 1.0
**Maintained By:** ShortStayNG Security Team
